<?php

// zim_base.php - ZIM_Base Class and full Docs (above methods)

// ZIM BASE - Database Simplified
// connection, insert, update, select, delete, makeVars, query, reply, lock
// https://zimjs.com/base/
// https://zimjs.com/base/zim_base.zip
// Free to use - donations welcome - https://zimjs.com/donate/

// GLOBAL SETTINGS
// Set defaults here to obscure sensitive data from main script.
// We operate in one Database and set this only once.
// Can override in main script if desired - just set:
// $database = "current_database", etc.
// before requiring the zim_base.php file.

// // Uncomment this code to use with YOUR ZIM Base!
// $server = (isset($server)) ? $server : "localhost";
// $username = (isset($username)) ? $username : "your_username";
// $password = (isset($password)) ? $password : "your_password";
// $database = (isset($database)) ? $database : "your_databasename";
// $charset = (isset($charset)) ? $charset : "utf8mb4";

// This is ZIM connection data so delete this and uncomment above
require_once('../do_not_use.php');

$error = NULL;
$errorCode = NULL;

// MYSQLi OBJECT CONNECTION
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
try {
    $mysqli = new mysqli($server, $username, $password, $database);
    $mysqli->set_charset($charset);
} catch (Exception  $e) {
    $mysqli = new stdClass();
    // $mysqli->error = $e->getMessage();
    // $mysqli->errorCode = $e->getCode();
    $error = $e->getMessage();
    $errorCode = $e->getCode();
}
unset($server, $username, $password, $database, $charset);


// ZIM SETTINGS
// ZIM is a JavaScript Canvas Framework at https://zimjs.com
// where you can easily code creativity!
// ZIM Base can be used with or without ZIM.
// Setting $zim to false will prevent the making of a half dozen global variables
// that ZIM uses to handle binding and reply headers for async calls.
// These are harmless - but if never using ZIM, set this to false.
// See inside the ZIM_Base class for more information on the following globals:
// ZIM GLOBALS: $type, $master, $data, $command, $extra, $lock, $unique

// See https://codepen.io/danzen/pen/oNjgMNX - for an example using ZIM
// See the hero.php in the zim_base.zip for the backend for this example
// See https://zimjs.com/docs.html?item=Bind for Bind() docs in ZIM
// See https://zimjs.com/docs.html?item=async for async() docs in ZIM
// See https://zimjs.com/docs.html?item=Ajax for Ajax() docs in ZIM

$zim = (isset($zim)) ? $zim : true;

// ZIM BASE COMMANDS
// https://zimjs.com/base/commands.php
// The ZIM_Base class uses PHP and MySQLi for safe MySQL database operations.
// The methods wrap SQL queries to reduce code to
// half the size of old MySQL and a third the size of MySQLi.
// The parameter values are all strings, arrays or associative arrays.
// This helps give database access a more consistent PHP feel.

// See https://zimjs.com/skool/lesson09.html for Data Theory and explanation
// See https://zimjs.com/learnjavascript.html#lesson09 for video support

// CLASS:
// $base = new ZIM_Base(mysqli, error, errorCode, zim); // runs at the bottom of this file

// METHODS:
// $base->makeVars(variable1, variable2, varliable3, etc.) - not needed with ZIM Bind
// $base->simplify(coupled) - for breaking up ZIM Bind variables if not storing as JSON
// $base->select(table, fields, where, more)
// $base->insert(table, fields, update, where, more)
// $base->update(table, update, where, more)
// $base->delete(table, where)
// $base->query(SQL) - returns result of general query (not a prepared statement so no binding)
// $base->makeVars(variable1, variable2, etc.) - from either GET OR POST
// $base->reply(messageType, message) - reply for ZIM Bind() / ZIM async() / ZIM Ajax()
// $base->setLock(table, id, lockid, idfield, lockfield) - provides record locking

// select(), insert(), update(), and delete() return a result object:
// $result->query - the SQL statement
// $result->names - array of field names that will be bound
// $result->types - array of types for the bound fields
// $result->values - array of values for the bound fields
// $result->success - true or false (opposite of error)
// $result->error - true or false (opposite of success)
// $result->native - reference to the raw MySQLi result object
// $result->metadata - information about the fields - eg. below
//      $result->metadata->fieldCount - count of fields
//      $result->metadata->fetch_field()->name - get next field name
//      $result->metadata->field_seek(0) // set fetch field index to 0 to start fetch again
// -- also these depending on method:
// $result->num_rows - number of rows selected
// $result->affected_rows - the number of rows inserted, updated or deleted
// $result->assoc - array of associative arrays matching selected fields eg. $result->assoc[1]["data"]
// $result->array - array of arrays in order of matching fields eg. $result->assoc[1][1]
// $result->record - associative array of first record eg. $result->record["data"]
// $result->row - array of first record eg. $result->row[1]
// $result->json - a single json object - assumes one field selected - and this is from the first result
// $result->json_array - json array of former json objects - assumes one json field was selected - decodes, adds to array and encodes

// See methods below for more information

class ZIM_Base {

    public $mysqli;
    public $type; // these are available as globals if zim is true
    public $master;
    public $data;
    public $command;
    public $extra;
    public $lock;
    public $unique;
    public $error;
    public $errorCode;

    public function __construct($mysqli, $error, $errorCode, $zim=true) {
        $this->mysqli = $mysqli;

        $this->error = $error;
        $this->errorCode = $errorCode;
       
        if ($zim) {
            global $type, $master, $data, $command, $extra, $lock, $unique, $header;

            // HEADER
            // A JS header is needed for JSONp with GET - harmless in POST with AJAX
            // however this will show raw HTML if using PHP to show an HTML result page
            // so there is a $header test at the bottom of this file
            // that checks for an $r variable which is a random cache busting number
            // that is sent by all ZIM operations.
            // If $r is sent then we set the header() here.
            if ($header) header('Content-type: text/javascript');

            // METHOD
            // We have arranged our ZIM Bind code to send the type of connection
            // This is placed on the URL so it gets collected with with GET regardless of type
            $this->type = $type = isset($_GET["type"]) ? $_GET["type"] : "get";

            // DATA
            // We collect get or post data as $_GET["property"] or $_POST["property"]
            // These are global environment variables in PHP
            // Data can come from an HTML form, JSONp call, AJAX call, etc. it does not matter
            // We use the PHP isset() function to find out if the property exists
            // and if not, then we set a default - here we set the default "get"
            // This code is the Ternary operator common in most languages
            // It is like an if statement all in one line
            // if there is a type then assign the type to $type else assign "get" to $type

            // ZIM Bind methods send the following variables
            // We may not use them all but it is handy to just copy this full code block
                // master - information for all "to" and "from" - like a room, etc.
                // data - the JSON data to store or the data requested
                // command - "to" or "from" to store or retrieve data
                // extra - holds extra information we might need like an id
                // lock - are we wanting to record lock - see ZIM PHP setLock() function
                // unique - should "to" unique? (if not, we can insert with update)

            $inputs = ["master", "data", "command", "extra", "lock", "unique"];
            foreach($inputs as $input) {
                if ($this->type == "get") {
                    $this->$input = $$input = isset($_GET[$input]) ? $_GET[$input] : "";
                } else {
                    $this->$input = $$input = isset($_POST[$input]) ? $_POST[$input] : "";
                }
            }

        } // end zim

    } // end constructor

    // ~~~~~~~~~~~~~~
    // PRIVATE FUNCTIONS - public functions will be of more interest - they are down below...


    private function getWhere($where) {
        $names = [];
        $vals = [];
        $types = [];
        if (isset($where)) {
            $string = " WHERE ";
            $count = 0;
            foreach ($where as $n=>$v) {
                if (gettype($v)=="array") {
                    // [prefix, infix, value]
                    if (count($v) == 1) {
                        $string .=  ($count==0?"":" ") .$v[0];
                        continue;
                    }                    
                    if ($v[0] == "OR" && is_array($v[2])) {
                        $v[0] = "REGEXP"; 
                        $pre = $count==0?"":" AND ";
                        $added = [];       
                        foreach ($v[2] as $item) {
                            array_push($added, "?");
                            array_push($vals, $item);
                            array_push($types, gettype($item)=="integer"?"i":"s");
                        }                
                        $names[$count] = "$n REGEXP CONCAT(".implode(',"|",',$added).")";                        
                    } else {   
                        $pre = $v[0] ? " $v[0] " : ($count==0?"":" AND ");   
                        if (!$v[1] || $v[1]=="") $v[1]="=";
                        array_push($names, "$n".$v[1]."?");
                        array_push($vals, $v[2]);
                        array_push($types, gettype($v)=="integer"?"i":"s");
                    }                    
                } else {
                    $pre = $count==0?"":" AND ";
                    array_push($names, "$n=?");
                    array_push($vals, $v);
                    array_push($types, gettype($v)=="integer"?"i":"s");
                }                
                $string .= $pre . $names[$count];
                $count++;
            }
        } else {
            $string = "";
        }
        return [$names, $vals, $types, $string];
    }

    // private function getWhere($where) {
    //     $names = [];
    //     $vals = [];
    //     $types = [];
    //     if (isset($where)) {
    //         $string = " WHERE ";
    //         $count = 0;
    //         foreach ($where as $n=>$v) {
    //             if (gettype($v)=="array") {
    //                 // [prefix, infix, value]
    //                 if (count($v) == 1) {
    //                     $string .=  ($count==0?"":" ") .$v[0];
    //                     continue;
    //                 }
    //                 $pre = $v[0] ? " $v[0] " : ($count==0?"":" AND ");
    //                 if (!$v[1] || $v[1]=="") $v[1]="=";
    //                 array_push($names, "$n".$v[1]."?");
    //                 array_push($vals, $v[2]);
    //             } else {
    //                 $pre = $count==0?"":" AND ";
    //                 array_push($names, "$n=?");
    //                 array_push($vals, $v);
    //             }
    //             array_push($types, gettype($v)=="integer"?"i":"s");
    //             $string .= $pre . $names[$count];
    //             $count++;
    //         }
    //     } else {
    //         $string = "";
    //     }
    //     return [$names, $vals, $types, $string];
    // }

    private function getField($fields) {
        $names = [];
        $binds = [];
        $vals = [];
        $types = [];

        foreach ($fields as $n=>$v) {
            array_push($names, $n);
            if (preg_match('/NOW/',$v)) {
                array_push($binds, $v);
            } else if ($v===null) {
                array_push($binds, "''");
            } else {
                array_push($binds, "?");
                array_push($vals, $v);
                array_push($types, gettype($v)=="integer"?"i":"s");
            }
        }
		
        if (count($fields) == 0) $string = "";
        else $string = "(".join(", ",$names).") VALUES (".join(", ",$binds).")";
        return [$names, $vals, $types, $string];
    }

    private function getUpdate($update, $keyword="UPDATE") {
        $names = [];
        $binds = [];
        $vals = [];
        $types = [];
        $pairs = [];
        $count = 0;
		$string = "";
		if (!is_null($update)) {
			foreach ($update as $n=>$v) {
				array_push($names, $n);
                if (($v[0] ?? null) == "\\") {
                    array_push($binds, ltrim($v, "\\"));
                } else if ($v=="NOW()") {
					array_push($binds, $v);
				} else if ($v===null) {
					array_push($binds, "''");
				} else {
					array_push($binds, "?");
					array_push($vals, $v);
					array_push($types, gettype($v)=="integer"?"i":"s");
				}
				array_push($pairs, $n."=".$binds[$count]);
				$count++;
			}
			if (count($update) == 0) $string = "";
			else $string = "$keyword " . join(", ",$pairs);
		}
        // echo $string;
        return [$names, $vals, $types, $string];
    }

    // ~~~~~~~~~~~~~~
    // PUBLIC FUNCTIONS

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // makeVars(variable1, variable2, varliable3, etc.)
    // makeVars(string, string, string, etc.)
    // DESCRIPTION
    // Gets either $_GET or $_POST variables and makes global variables from them
    // Automatically detects type
    // EXAMPLES
    // $base->makeVars("id", "data");
    // echo $id; // id
    // echo $data; // data
    public function makeVars() {
        $numargs = func_num_args();
        $args = func_get_args();
        for ($i=0; $i<$numargs; $i++) {
            $arg = $args[$i];
            global $$arg;
            if (isset($_POST[$arg])) $$arg = $_POST[$arg];
            else if (isset($_GET[$arg])) $$arg = $_GET[$arg];
        }
        return $numargs;
    }
    
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // simplify(coupled)
    // simplify(boolean-default false)
    // DESCRIPTION
    // JSON decodes and converts ZIM Bind data object to associative array of properties and values 
    // imagine that we have data {"circle":{"x":"10", "y":"20"},"count":{"currentValue":"0"}}
    // if coupled is set to true then simplify() returns and associative array:
    //      ["circle_x"=>"10", "circle_y"=>"20", "count_currentValue"=>"0"]
    // if coupled is left off or set to false then will return data based on bindIDs and ignore bind property name 
    // this could lose data if the bindID has more than one property - here we lose the x data
    //      ["circle"=>"20", count=>"0"]
    // but often storing form data we have only one bit of data per bindID:
    // for example {"name":{"text":"Dan Zen"},"occupation":{"text":"Inventor"}}
    // with coupled:
    //      ["name_text"=>"Dan Zen", "occupation_text"=>"Inventor"]
    // without coupled:
    //      ["name"=>"Dan Zen", "occupation"=>"Inventor"]
    // The resulting associative array can be merged with other desired data and passed to $bind->insert() etc.
    // see the example
    
    // EXAMPLES
    // $pairs = ["uid"=>$master, "date"=>"NOW()"];
    // $pairs = array_merge($pairs, $data->simplify());
    // $result = $base->insert("zim_survey", $pairs);
    public function simplify($coupled=null) {
        $arr = array();
        $d = json_decode($this->data);
        foreach ($d as $key=>$value) {
            // data may already be paired if paired parameter in ZIM Bind is set to true
            // if it is not paired then $value will be another object
            if (is_object($value)) {                
                foreach ($value as $k=>$v) {                    
                    if ($coupled) { // this will couple the bindID and property name
                        $arr[$key."_".$k] = $v;                         
                    } else { // this will store the last data for a key in the key - (no property name)                        
                        $arr[$key] = $v;     
                    }                
                }
            } else { // data is already paired
                $couples = explode("_", $key);
                if (count($couples)>1 && !$coupled) {
                    $k = array_shift($couples);
                    $arr[$k] = $value; 
                } else {
                    $arr[$key] = $value; 
                }
            }
        }
        return $arr;
    }   
    
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // select(table, fields, where, more)
    // select(string, string or array, assoc with value or [prefix, infix, value], string)
    // EXAMPLES
    // $result = $base->select("zim_hero", "name", ["name"=>$name]);
    // $result = $base->select("zim_hero", ["name","age"], ["name"=>$name, age=>$age]);
    // $result = $data->select("zim_hero", ["name","age"], ["name"=>["","!=",$name], "age"=>["OR",null,$age]], "ORDER BY count DESC LIMIT 10");
    // $result = $data->select("zim_hero", ["name","age"], ["age"=>["OR",null,[5,10,15]]]); // or will use REGEXP to find any of these values
    // $result = $base->select("zim_users, zim_hero", "*", ["zim_users.name"=>$name, "zim_hero.name"=>$name], "ORDER BY name"); // join
    // $result = $data->select("zim_hero", *, null, "WHERE record='first'"); // override binding by adding where to more
    // $result = $data->select("zim_hero", "*");
    // $result = $data->select("zim_hero"); // or same as above
    // result OBJECT METHODS AND PROPERTIES
    // $result->query - the SQL statement
    // $result->names - array of field names that will be bound
    // $result->types - array of types for the bound fields
    // $result->values - array of values for the bound fields
    // $result->success - true or false (opposite of error)
    // $result->error - true or false (opposite of success)
    // $result->num_rows - number of rows selected
    // $result->native - the original mysqli result
    // $result->metadata - information about the fields:
    //      $result->metadata->field_count // number of fields
    //      $result->metadata->fetch_field() // gets the next field
    //          includes following properties (from PHP.net):
    //          ->name 	The name of the column
    //          ->orgname 	Original column name if an alias was specified
    //          ->table 	The name of the table this field belongs to (if not calculated)
    //          ->orgtable 	Original table name if an alias was specified
    //          ->def 	Reserved for default value, currently always ""
    //          ->db 	Database
    //          ->catalog 	The catalog name, always "def"
    //          ->max_length 	The maximum width of the field for the result set.
    //          ->length 	The width of the field, as specified in the table definition.
    //          ->charsetnr 	The character set number for the field.
    //          ->flags 	An integer representing the bit-flags for the field.
    //          ->type 	The data type used for this field
    //          ->decimals 	The number of decimals used (for integer fields)
    //      $result->metadata->fetch_field_direct($index) // gets field at an index
    //      $result->metadata->fetch_fields() // gives object of fields
    //      $result->metadata->field_seek($index) // sets seek too an index - set to 0 to start fetch again
    //      $result->metadata->field_tell() // gets seek index
    // $result->assoc - array of associative arrays matching selected fields eg. $result->assoc[1]["data"]
    // $result->record - the associative array of the first result eg. $result->record["data"]
    // $result->array - array of arrays in order of matching fields eg. $result->assoc[1][0]
    // $result->row - array of first record eg. $result->row[1]
    // $result->json - a single json object - assumes one filed selected - and this is from the first result
    // $result->json_array - json array of former json objects - assumes one json field was selected - decodes, adds to array and encodes
    public function select($table, $fields="*", $where=null, $more=null) {

        if ($fields == "*" || $fields == null) {
            $fieldString = "*";
        } else if (gettype($fields) == "array") {
            $fieldString = join(", ", $fields);
        } else {
            $fieldString = $fields;
        }
        list($names, $vals, $types, $whereString) = $this->getWhere($where);
        $more = isset($more)?$more:"";
        $query = "SELECT " . $fieldString . " FROM " . $table . $whereString . " " . $more;

        $obj = new stdClass();
        $obj->query = $query;
        if (isset($names)) $obj->names = $names;
        if (isset($types)) $obj->types = $types;
        if (isset($vals)) $obj->values = $vals;

        $stmt =  $this->mysqli->stmt_init();
        if ($stmt->prepare($query)) {
			if (count($vals) > 0)
				$stmt->bind_param(join("", $types), ...$vals);
            $stmt->execute();
            // $obj->record = $stmt->get_result()->fetch_assoc();
            $result = $stmt->get_result();
            $obj->metadata = $stmt->result_metadata();
            $stmt->close();
            $obj->num_rows = $result->num_rows;
            $obj->native = $result;
            $obj->assoc = $result->fetch_all(MYSQLI_ASSOC);
			$obj->json = "{}";
			if (count($obj->assoc) > 0) {
				$obj->record = $obj->assoc[0];
				foreach($obj->record as $key=>$val) {
					$obj->json = $val; // assumes first field
					break;
				}
			}
            $result->data_seek(0);
            $obj->array = $result->fetch_all(MYSQLI_NUM);
            $arranged = [];
			if (count($obj->array) > 0) {
				$obj->row = $obj->array[0];
				foreach($obj->array as $d) {
					array_push($arranged, json_decode($d[0]));
				}
			}
            $obj->json_array = json_encode($arranged);
            $obj->success = true;
            $obj->error = false;
        } else {
            $obj->error = true;
            $obj->success = false;
        }
        return $obj;
    } // end select

    public function selectTest($table, $fields="*", $where=null, $more=null) {

        if ($fields == "*" || $fields == null) {
            $fieldString = "*";
        } else if (gettype($fields) == "array") {
            $fieldString = join(", ", $fields);
        } else {
            $fieldString = $fields;
        }
        list($names, $vals, $types, $whereString) = $this->getWhere($where);
        $more = isset($more)?$more:"";
        $query = "SELECT " . $fieldString . " FROM " . $table . $whereString . " " . $more;

        echo $query;

    } // end select


    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // insert(table, fields, update, where, more)
    // insert(string, assoc array, assoc array, assoc with value or [prefix, infix, value], string)
    // EXAMPLES
    // $result = $base->insert("zim_hero", ["name"=>$name, "data"=>$data, "date"=>"NOW()"], ["data"=>$data]);
    // see insert() for example of $where and $more
    // result OBJECT METHODS AND PROPERTIES
    // $result->query - the SQL statement
    // $result->names - array of field names that will be bound
    // $result->types - array of types for the bound fields
    // $result->values - array of values for the bound fields
    // $result->affected_rows - the number of rows inserted
    // $result->success - true or false (opposite of error)
    // $result->error - true or false (opposite of success)

    public function insert($table, $fields, $update=null, $where=null, $more=null) {

        // $query = "INSERT INTO zim_hero (name, data, date) VALUES (?, ?, NOW()) ON DUPLICATE KEY UPDATE data = ?, name =?";

        list($fieldNames, $fieldVals, $fieldTypes, $fieldString) = $this->getField($fields);
        list($whereNames, $whereVals, $whereTypes, $whereString) = $this->getWhere($where);
        $more = isset($more)?$more:"";
        list($updateNames, $updateVals, $updateTypes, $updateString) = $this->getUpdate($update);

        if ($updateString) $updateString = " ON DUPLICATE KEY " . $updateString;
        $query = "INSERT INTO " . $table ." ". $fieldString ." ". $whereString ." ". $more ." ". $updateString;
        
        $types = array_merge($fieldTypes, $whereTypes, $updateTypes);
        $vals = array_merge($fieldVals, $whereVals, $updateVals);

        $obj = new stdClass();
        $obj->query = $query;
		if (isset($names)) $obj->names = $names;
        if (isset($types)) $obj->types = $types;
        if (isset($vals)) $obj->values = $vals;
            
        $stmt = $this->mysqli->stmt_init();
        if ($stmt->prepare($query)) {  
            $stmt->bind_param(join("",$types), ...$vals);
            $stmt->execute();
            $obj->affected_rows = $stmt->affected_rows;
            $stmt->close();
            $obj->success = true;
            $obj->error = false;
        } else {
            $obj->error = true;
            $obj->success = false;
        }
        
        
                
        return $obj;
    } // end insert


    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // update(table, update, where, more)
    // insert(string, assoc array, assoc with value or [prefix, infix, value], string)
    // EXAMPLES
    // $result = $base->update("zim_hero", ["data"=>$data, "date"=>"NOW()"], ["name"=>$name]);
    // result OBJECT METHODS AND PROPERTIES
    // $result->query - the SQL statement
    // $result->names - array of field names that will be bound
    // $result->types - array of types for the bound fields
    // $result->values - array of values for the bound fields
    // $result->affected_rows - the number of rows updated
    // $result->success - true or false (opposite of error)
    // $result->error - true or false (opposite of success)

    public function update($table, $update, $where=null, $more=null) {

        // $query = "UPDATE zim_hero SET data=?, date=NOW() WHERE name=?";

        list($updateNames, $updateVals, $updateTypes, $updateString) = $this->getUpdate($update, "SET");
        list($whereNames, $whereVals, $whereTypes, $whereString) = $this->getWhere($where);
        $more = isset($more)?$more:"";

        $query = "UPDATE " . $table ." ". $updateString ." ". $whereString ." ". $more;
      
        $types = array_merge($updateTypes, $whereTypes);
        $vals = array_merge($updateVals, $whereVals);

        $obj = new stdClass();
        $obj->query = $query;
        if (isset($names)) $obj->names = $names;
        if (isset($types)) $obj->types = $types;
        if (isset($vals)) $obj->values = $vals;

        $stmt = $this->mysqli->stmt_init();
        if ($stmt->prepare($query)) {
            $stmt->bind_param(join("",$types), ...$vals);
            $stmt->execute();
            $obj->affected_rows = $stmt->affected_rows;
            $stmt->close();
            $obj->success = true;
            $obj->error = false;
        } else {
            $obj->error = true;
            $obj->success = false;
        }
        return $obj;
    } // end update


    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // delete(table, where)
    // delete(string, array)
    // EXAMPLES
    // $result = $base->delete("zim_hero", ["name"=>$name]);
    // $result = $data->delete("zim_hero", ["name"=>["NOT","=",$name]]);
    // result OBJECT METHODS AND PROPERTIES
    // $result->query - the SQL statement
    // $result->names - array of field names that will be bound
    // $result->types - array of types for the bound fields
    // $result->values - array of values for the bound fields
    // $result->affected_rows - the number of rows deleted
    // $result->success - true or false (opposite of error)
    // $result->error - true or false (opposite of success)
    public function delete($table, $where=null) {

        list($names, $vals, $types, $whereString) = $this->getWhere($where);
        $query = "DELETE FROM " . $table . $whereString;

        $obj = new stdClass();
        $obj->query = $query;
        if (isset($names)) $obj->names = $names;
        if (isset($types)) $obj->types = $types;
        if (isset($vals)) $obj->values = $vals;

        $stmt =  $this->mysqli->stmt_init();
        if ($stmt->prepare($query)) {
            $stmt->bind_param(join("",$types), ...$vals);
            $stmt->execute();
            $obj->affected_rows = $stmt->affected_rows;
            $stmt->close();
            $obj->success = true;
            $obj->error = false;
        } else {
            $obj->error = true;
            $obj->success = false;
        }
        return $obj;
    } // end delete


    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // query(SQL)
    // query(string)
    // EXAMPLES
    // $result = $base->query("TRUNCATE TABLE zim_base");
    // $result->query - the SQL statement
    // $result->native - the original mysqli result
    // $result->success - true or false (opposite of error)
    // $result->error - true or false (opposite of success)
    public function query($sql) {
        $result = $this->mysqli->query($sql);
        $obj = new stdClass();
        $obj->query = $sql;
        $obj->native = $result;
        if ($result) {
            $obj->success = true;
            $obj->error = false;
        } else {
            $obj->error = true;
            $obj->success = false;
        }
        return $obj;
    }


    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // reply(messageType, message)
    // reply(string, string)
    // EXAMPLES
    // if ($name == "") $base->reply("error","missing id");
    // or
    // $base->reply($result->json);
    public function reply($messageType, $message=null) {
        $type = $this->type; // do not rely on globals as they could be turned off
        $command = $this->command;
        // for JSONp, each command has a different callback function
        // we could dynamically create the function names but we will keep it simple
        if ($message) {
            $message = json_encode([$messageType=>$message]);
        } else {
            $message = $messageType;
        }
        if ($command == "to") echo $type == "get" ? "async.callbackTo($message)" : $message;
        else if ($command == "fromTo") echo $type == "get" ? "async.callbackFromTo($message)" : $message;
        else if ($command == "from") echo $type == "get" ? "async.callbackFrom($message)" : $message;
        else if ($command == "removeAll") echo $type == "get" ? "async.callbackRemoveAll($message)" : $message;
        else echo $message;
        exit; // should only ever send one message back
    }


    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // setLock(table, id, lockid, idfield, lockfield)
    // setLock(string, string, string, string, string)
    // EXAMPLE
    // if ($lockid != "") $base->setLock("zim_base", $id, $lockid);
    // DESCRIPTION
    // Provides record locking functionality
    // handy when multiple clients are sharing same json data for instance
    // Waits for unlocked record then sets lock
    // Lock should be cleared when successfully updated
    // WARNING Will be auto overwritten after 1 second

    // If desired, the idfield and lockfield names can be set
    // by default they are "id" and "lockid" field names in the table
    // setLock(table, idvalue, lockvalue, idfield, lockfield);
    // returns true (1) or false (0)

    // Note used with ZIM Bind() toLock()
    // which collects current data and sets lock
    // sends data back to client which then appends or edits data
    // and then sets the new data on the table and clears the lock field
    public function setLock($table, $id, $lockid, $idfield="id", $lockfield="lockid") {
        $mysqli = $this->mysqli;
        $tries = 0;
		$ready = 0;
        while (!$ready) {
            // $table, $idfield and $lockfield variables are set by programmer in PHP - not by user
            $query = "SELECT $lockfield FROM $table WHERE $idfield=? AND $lockfield IS NOT NULL AND $lockfield != ''";
            $stmt = $mysqli->stmt_init();
            if ($stmt->prepare($query)) {
                $stmt->bind_param("s", $id);
                $stmt->execute();
                $stmt->store_result();
                if ($stmt->num_rows == 0) {
                    $ready = 1; // no locked record
                } else if ($tries > 10) {
                    $ready = 1; // record is clogged - may want to send error?
                } else {
                    $tries++;
                    sleep(.1); // wait 100ms for the next try
                }
                $stmt->close();
            } else {
                $ready = 1;
            }
        }
        // set the lock
        $query = "INSERT INTO $table ($idfield, $lockfield) VALUES (?, ?) ON DUPLICATE KEY UPDATE $lockfield = ?";
        $stmt = $mysqli->stmt_init();
        if ($stmt->prepare($query)) {
            $stmt->bind_param("sss", $id, $lockid, $lockid);
            $stmt->execute();
            return 1;
        } else {
            return 0;
        }
    }


} // ZIM_Base

// ZIM sends random number (r) to defeat cache
// use this to determine ZIM_Base default header setting
if (isset($_GET["r"])) $header = true;

// Create the ZIM_Base object
$base = new ZIM_Base($mysqli, $error, $errorCode, $zim);


?>
