import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;
import java.sql.Connection;
import java.sql.Date;
import java.sql.SQLException;


public class AuthorDB {
    private final Connection connection;

    public AuthorDB(Connection connection) {
        this.connection = connection;
    }

    public void insert(Author author) throws DBException {
        if (author == null) {
            return;
        }

        var sql = "INSERT INTO Authors (FirstName, LastName, BirthDate) VALUES (?,?,?)";

        try (var statement = connection.prepareStatement(sql)) {
            // Bind values to parameters
            statement.setString(1, author.getFirstName());
            statement.setString(2, author.getLastName());
            statement.setDate(3, Date.valueOf(author.getBirthDate()));

            // Execute the query
            statement.executeUpdate();

        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public void insertMany(List<Author> authors) throws DBException {

        if (authors == null || authors.isEmpty()) {
            return;
        }

        var sql = "INSERT INTO Authors (FirstName, LastName, BirthDate) VALUES (?, ?, ?)";

        try (var statement = connection.prepareStatement(sql)) {
            for (Author author : authors) {

                // Bind values
                statement.setString(1, author.getFirstName());
                statement.setString(2, author.getLastName());
                statement.setDate(3, Date.valueOf(author.getBirthDate()));

                // Add to batch
                statement.addBatch();
            }

            // Execute batch
            statement.executeBatch();
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    private Author createAuthorFrom(ResultSet resultset) throws SQLException {
        return new Author(
                resultset.getInt("AuthorID"),
                resultset.getString("FirstName"),
                resultset.getString("LastName"),
                resultset.getDate("BirthDate").toLocalDate()
        );
    }

    public List<Author> findAll() throws DBException {
        var sql = "SELECT AuthorID, FirstName, LastName, BirthDate FROM Authors";

        var authors = new ArrayList<Author>();

        try (var statement = connection.createStatement();
             var resultset = statement.executeQuery(sql)) {
            //
            while (resultset.next()) {
                // create the author object
                authors.add(createAuthorFrom(resultset));
            }

            return authors;
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public ArrayList<Author> find(int limit, int offset) throws DBException {

        var sql = "SELECT AuthorID, FirstName, LastName, BirthDate " +
                  "FROM Authors   " +
                  "ORDER BY AuthorID " +
                  "OFFSET @offset ROWS FETCH FIRST @limit ROWS ONLY";

        var authors = new ArrayList<Author>();

        try (var statement = connection.prepareStatement(sql)) {
            // Set values to parameters
            statement.setInt(1, offset);
            statement.setInt(1, limit);

            try(var resultset = statement.executeQuery()) {
                while (resultset.next()) {
                    authors.add(createAuthorFrom(resultset));
                }
            }

            return authors;

        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }



    public Author findById(int authorId) throws DBException {

        var sql = "SELECT AuthorID, FirstName, LastName, BirthDate FROM Authors WHERE AuthorID = ?";

        try (var statement = connection.prepareStatement(sql)) {
            // Bind values to parameters
            statement.setInt(1, authorId);

            // Execute the query
            try (var resultset = statement.executeQuery()) {
                //
                if (!resultset.next()) {
                    throw new DBException("The author with id " + authorId + " not found.");
                }

                // create the author object
                return createAuthorFrom(resultset);
            }

        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

}
