# Copyright 2025 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# [START storage_control_disable_anywhere_cache]
require "google/cloud/storage/control"

# Disables a specific Anywhere Cache instance for a bucket.
# This operation disables the cache but does not delete it. A disabled cache
# can be re-enabled later.
#
# @param bucket_name [String] The name of the bucket.
# @param anywhere_cache_id [String] A value that, along with the bucket's
#   name, uniquely identifies the cache instance (e.g., "us-east1-b").
#
# @example
#   disable_anywhere_cache(
#     bucket_name: "your-unique-bucket-name",
#     anywhere_cache_id: "us-east1-b"
#   )
#
def disable_anywhere_cache bucket_name:, anywhere_cache_id:
  # Create a client object. The client can be reused for multiple calls.
  storage_control_client = Google::Cloud::Storage::Control.storage_control
  # Set project to "_" to signify global bucket
  parent = "projects/_/buckets/#{bucket_name}"
  name = "#{parent}/anywhereCaches/#{anywhere_cache_id}"

  # Create a request.
  request = Google::Cloud::Storage::Control::V2::DisableAnywhereCacheRequest.new(
    name: name
  )
  # The request disables the cache, but does not delete it.
  # The cache can be re-enabled later.
  begin
    result = storage_control_client.disable_anywhere_cache request
    puts "Successfully #{result.state&.downcase} anywhereCache - #{result.name}."
  rescue Google::Cloud::Error => e
    puts "Failed to disable AnywhereCache. Error: #{e.message}"
  end
end
# [END storage_control_disable_anywhere_cache]
disable_anywhere_cache bucket_name: ARGV.shift, anywhere_cache_id: ARGV.shift if $PROGRAM_NAME == __FILE__
