# Copyright 2025 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require_relative "helper"
require_relative "../storage_control_create_anywhere_cache"
require_relative "../storage_control_list_anywhere_caches"
require_relative "../storage_control_get_anywhere_cache"
require_relative "../storage_control_update_anywhere_cache"
require_relative "../storage_control_pause_anywhere_cache"
require_relative "../storage_control_resume_anywhere_cache"
require_relative "../storage_control_disable_anywhere_cache"

describe "Storage Control Anywhere Cache" do
  let(:bucket_name) { random_bucket_name }
  let(:zone) { "us-east1-b" }
  # Set project to "_" to signify global bucket
  let(:anywhere_cache_name) { "projects/_/buckets/#{bucket_name}/anywhereCaches/#{zone}" }

  before :all do
    create_bucket_helper bucket_name
  end

  after :all do
    count_anywhere_caches bucket_name # Ensure all caches are deleted before deleting bucket
    delete_bucket_helper bucket_name
  end

  it "handles Anywhere cache lifecycle in sequence" do
    out_create, _err = capture_io do
      create_anywhere_cache bucket_name: bucket_name, zone: zone
    end
    assert_includes out_create, "Successfully created anywhereCache - #{anywhere_cache_name}."

    out_list, _err = capture_io do
      list_anywhere_caches bucket_name: bucket_name
    end
    assert_includes out_list, "AnywhereCache #{anywhere_cache_name} found in list."

    out_get, _err = capture_io do
      get_anywhere_cache bucket_name: bucket_name, anywhere_cache_id: zone
    end
    assert_includes out_get, "Successfully fetched anywhereCache - #{anywhere_cache_name}."

    out_update, _err = capture_io do
      update_anywhere_cache bucket_name: bucket_name, anywhere_cache_id: zone
    end
    assert_includes out_update, "Successfully updated anywhereCache - #{anywhere_cache_name}."

    out_pause, _err = capture_io do
      pause_anywhere_cache bucket_name: bucket_name, anywhere_cache_id: zone
    end
    assert_includes out_pause, "Successfully paused anywhereCache - #{anywhere_cache_name}."

    out_resume, _err = capture_io do
      resume_anywhere_cache bucket_name: bucket_name, anywhere_cache_id: zone
    end
    assert_includes out_resume, "Successfully resumed anywhereCache - #{anywhere_cache_name}."

    out_disable, _err = capture_io do
      disable_anywhere_cache bucket_name: bucket_name, anywhere_cache_id: zone
    end
    assert_includes out_disable, "Successfully disabled anywhereCache - #{anywhere_cache_name}."
  end
end
