# Authentication

In general, the google-cloud-pubsub library uses [Service
Account](https://cloud.google.com/iam/docs/creating-managing-service-accounts)
credentials to connect to Google Cloud services. When running within [Google
Cloud Platform environments](#google-cloud-platform-environments)
the credentials will be discovered automatically. When running on other
environments, the Service Account credentials can be specified by providing the
path to the [JSON
keyfile](https://cloud.google.com/iam/docs/managing-service-account-keys) for
the account (or the JSON itself) in [environment
variables](#environment-variables). Additionally, Cloud SDK credentials can also
be discovered automatically, but this is only recommended during development.

## Quickstart

1. [Create a service account and credentials](#creating-a-service-account).
2. Set the [environment variable](#environment-variables).

```sh
export PUBSUB_CREDENTIALS=/path/to/json`
```

3. Initialize the client.

```ruby
require "google/cloud/pubsub"

client = Google::Cloud::PubSub.new
```

## Project and Credential Lookup

The google-cloud-pubsub library aims to make authentication
as simple as possible, and provides several mechanisms to configure your system
without providing **Project ID** and **Service Account Credentials** directly in
code.

**Project ID** is discovered in the following order:

1. Specify project ID in method arguments
2. Specify project ID in configuration
3. Discover project ID in environment variables
4. Discover GCE project ID
5. Discover project ID in credentials JSON

**Credentials** are discovered in the following order:

> [!WARNING]
> If you accept a credential configuration (JSON file or Hash) from an
> external source for authentication to Google Cloud, you must validate it before
> providing it to a Google API client library. Providing an unvalidated credential
> configuration to Google APIs can compromise the security of your systems and data.

1. Specify credentials in method arguments
2. Specify credentials in configuration
3. Discover credentials path in environment variables
4. Discover credentials JSON in environment variables
5. Discover credentials file in the Cloud SDK's path
6. Discover GCE credentials

### Google Cloud Platform environments

When running on Google Cloud Platform (GCP), including Google Compute Engine (GCE),
Google Kubernetes Engine (GKE), Google App Engine (GAE), Google Cloud Functions
(GCF) and Cloud Run, the **Project ID** and **Credentials** and are discovered
automatically. Code should be written as if already authenticated.

### Environment Variables

The **Project ID** and **Credentials JSON** can be placed in environment
variables instead of declaring them directly in code. Each service has its own
environment variable, allowing for different service accounts to be used for
different services. (See the READMEs for the individual service gems for
details.) The path to the **Credentials JSON** file can be stored in the
environment variable, or the **Credentials JSON** itself can be stored for
environments such as Docker containers where writing files is difficult or not
encouraged.

The environment variables that google-cloud-pubsub checks for project ID are:

1. `PUBSUB_PROJECT`
2. `GOOGLE_CLOUD_PROJECT`

The environment variables that google-cloud-pubsub checks for credentials
are configured on {Google::Cloud::PubSub::Credentials}:

1. `PUBSUB_CREDENTIALS` - Path to JSON file, or JSON contents
2. `PUBSUB_KEYFILE` - Path to JSON file, or JSON contents
3. `GOOGLE_CLOUD_CREDENTIALS` - Path to JSON file, or JSON contents
4. `GOOGLE_CLOUD_KEYFILE` - Path to JSON file, or JSON contents
5. `GOOGLE_APPLICATION_CREDENTIALS` - Path to JSON file

```ruby
require "google/cloud/pubsub"

ENV["PUBSUB_PROJECT"]     = "my-project-id"
ENV["PUBSUB_CREDENTIALS"] = "path/to/keyfile.json"

client = Google::Cloud::PubSub.new
```

### Configuration

The **Project ID** and the path to the **Credentials JSON** file can be configured
instead of placing them in environment variables or providing them as arguments.

```ruby
require "googleauth"
require "google/cloud/pubsub"

credentials = ::Google::Auth::ServiceAccountCredentials.make_creds(
  json_key_io: ::File.open("/path/to/keyfile.json")
)

Google::Cloud::PubSub.configure do |config|
  config.project_id  = "my-project-id"
  config.credentials = credentials
end

client = Google::Cloud::PubSub.new
```

### Cloud SDK

This option allows for an easy way to authenticate during development. If
credentials are not provided in code or in environment variables, then Cloud SDK
credentials are discovered.

To configure your system for this, simply:

1. [Download and install the Cloud SDK](https://cloud.google.com/sdk)
2. Authenticate using OAuth 2.0 `$ gcloud auth login`
3. Write code as if already authenticated.

**NOTE:** This is _not_ recommended for running in production. The Cloud SDK
*should* only be used during development.

## Creating a Service Account

Google Cloud requires a **Project ID** and **Service Account Credentials** to
connect to the APIs. You will use the **Project ID** and **JSON key file** to
connect to most services with google-cloud-pubsub.

If you are not running this client within [Google Cloud Platform
environments](#google-cloud-platform-environments), you need a Google
Developers service account.

1. Visit the [Google Cloud Console](https://console.cloud.google.com/project).
1. Create a new project or click on an existing project.
1. Activate the menu in the upper left and select **APIs & Services**. From
   here, you will enable the APIs that your application requires.

   *Note: You may need to enable billing in order to use these services.*

1. Select **Credentials** from the side navigation.

   Find the "Create credentials" drop down near the top of the page, and select
   "Service account" to be guided through downloading a new JSON key file.

   If you want to re-use an existing service account, you can easily generate 
   a new key file. Just select the account you wish to re-use click the pencil
   tool on the right side to edit the service account, select the **Keys** tab,
   and then select **Add Key**.

   The key file you download will be used by this library to authenticate API
   requests and should be stored in a secure location.

## Troubleshooting

If you're having trouble authenticating you can ask for help by following the
[Troubleshooting Guide](TROUBLESHOOTING.md).
