# Copyright 2016 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


##
# This file is here to be autorequired by bundler, so that the
# Google::Cloud.datastore and Google::Cloud#datastore methods can be available,
# but the library and all dependencies won't be loaded until required and used.


gem "google-cloud-core"
require "google/cloud" unless defined? Google::Cloud.new
require "google/cloud/config"
require "googleauth"

module Google
  module Cloud
    ##
    # Creates a new object for connecting to the Datastore service.
    # Each call creates a new connection.
    #
    # For more information on connecting to Google Cloud see the
    # {file:AUTHENTICATION.md Authentication Guide}.
    #
    # @param [String, Array<String>] scope The OAuth 2.0 scopes controlling the
    #   set of resources and operations that the connection can access. See
    #   [Using OAuth 2.0 to Access Google
    #   APIs](https://developers.google.com/identity/protocols/OAuth2).
    #
    #   The default scope is:
    #
    #   * `https://www.googleapis.com/auth/datastore`
    # @param [Integer] timeout Default timeout to use in requests. Optional.
    #
    # @return [Google::Cloud::Datastore::Dataset]
    #
    # @example
    #   require "google/cloud"
    #
    #   gcloud  = Google::Cloud.new
    #   datastore = gcloud.datastore
    #
    #   task = datastore.entity "Task" do |t|
    #     t["type"] = "Personal"
    #     t["done"] = false
    #     t["priority"] = 4
    #     t["description"] = "Learn Cloud Datastore"
    #   end
    #
    #   datastore.save task
    #
    # @example You shouldn't need to override the default scope, but you can:
    #   require "google/cloud"
    #
    #   gcloud  = Google::Cloud.new
    #   platform_scope = "https://www.googleapis.com/auth/cloud-platform"
    #   datastore = gcloud.datastore scope: platform_scope
    #
    def datastore scope: nil, timeout: nil, database_id: nil
      timeout ||= @timeout
      Google::Cloud.datastore @project, @keyfile,
                              scope: scope,
                              timeout: timeout,
                              database_id: database_id
    end

    ##
    # Creates a new object for connecting to the Datastore service.
    # Each call creates a new connection.
    #
    # For more information on connecting to Google Cloud see the
    # {file:AUTHENTICATION.md Authentication Guide}.
    #
    # @param [String] project_id Identifier for a Datastore project. If not
    #   present, the default project for the credentials is used.
    # @param [Google::Auth::Credentials] credentials A Google::Auth::Credentials
    #   object. (See {Datastore::Credentials})
    #   @note Warning: Passing a `String` to a keyfile path or a `Hash` of credentials
    #     is deprecated. Providing an unvalidated credential configuration to
    #     Google APIs can compromise the security of your systems and data.
    #
    #   @example
    #
    #     # The recommended way to provide credentials is to use the `make_creds` method
    #     # on the appropriate credentials class for your environment.
    #
    #     credentials = ::Google::Auth::ServiceAccountCredentials.make_creds(
    #       json_key_io: ::File.open("/path/to/keyfile.json")
    #     )
    #
    #     datastore = Google::Cloud::Datastore.new(
    #       project_id: "my-project-id",
    #       credentials: credentials
    #     )
    # @param [String, Array<String>] scope The OAuth 2.0 scopes controlling the
    #   set of resources and operations that the connection can access. See
    #   [Using OAuth 2.0 to Access Google
    #   APIs](https://developers.google.com/identity/protocols/OAuth2).
    #
    #   The default scope is:
    #
    #   * `https://www.googleapis.com/auth/datastore`
    # @param [Integer] timeout Default timeout to use in requests. Optional.
    #
    # @return [Google::Cloud::Datastore::Dataset]
    #
    # @example
    #   require "google/cloud"
    #
    #   datastore = Google::Cloud.datastore "my-todo-project",
    #                                       "/path/to/keyfile.json"
    #
    #   task = datastore.entity "Task", "sampleTask" do |t|
    #     t["type"] = "Personal"
    #     t["done"] = false
    #     t["priority"] = 4
    #     t["description"] = "Learn Cloud Datastore"
    #   end
    #
    #   datastore.save task
    #
    def self.datastore project_id = nil, credentials = nil, scope: nil,
                       timeout: nil, database_id: nil
      require "google/cloud/datastore"
      Google::Cloud::Datastore.new project_id: project_id,
                                   credentials: credentials,
                                   scope: scope,
                                   timeout: timeout,
                                   database_id: database_id
    end
  end
end

# Set the default bigquery configuration
Google::Cloud.configure.add_config! :datastore do |config|
  default_project = Google::Cloud::Config.deferred do
    ENV["DATASTORE_DATASET"] || ENV["DATASTORE_PROJECT"]
  end
  default_creds = Google::Cloud::Config.deferred do
    Google::Cloud::Config.credentials_from_env(
      "DATASTORE_CREDENTIALS", "DATASTORE_CREDENTIALS_JSON",
      "DATASTORE_KEYFILE", "DATASTORE_KEYFILE_JSON"
    )
  end
  default_emulator = Google::Cloud::Config.deferred do
    ENV["DATASTORE_EMULATOR_HOST"]
  end
  default_scopes = [
    "https://www.googleapis.com/auth/cloud-platform", "https://www.googleapis.com/auth/datastore"
  ]

  config.add_field! :project_id, default_project, match: String, allow_nil: true
  config.add_alias! :project, :project_id
  config.add_field! :credentials, default_creds, match: [String, Hash, Google::Auth::Credentials], allow_nil: true
  config.add_alias! :keyfile, :credentials
  config.add_field! :scope, default_scopes, match: [String, Array]
  config.add_field! :quota_project, nil, match: String
  config.add_field! :timeout, nil, match: Integer
  config.add_field! :emulator_host, default_emulator, match: String, allow_nil: true
  config.add_field! :endpoint, "datastore.googleapis.com", match: String
  config.add_field! :database_id, "", match: String
end
