/**
 * Copyright (c) 2015-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * @providesModule PickerIOS
 *
 * This is a controlled component version of RCTPickerIOS
 */
'use strict';

var NativeMethodsMixin = require('../../Renderer/shims/NativeMethodsMixin');
var React = require('../../react-native/React');
const PropTypes = require('prop-types');
var StyleSheet = require('../../StyleSheet/StyleSheet');
var StyleSheetPropType = require('../../StyleSheet/StyleSheetPropType');
var TextStylePropTypes = require('../../Text/TextStylePropTypes');
var View = require('../View/View');
const ViewPropTypes = require('../View/ViewPropTypes');
var processColor = require('../../StyleSheet/processColor');

var createReactClass = require('create-react-class');
var itemStylePropType = StyleSheetPropType(TextStylePropTypes);
var requireNativeComponent = require('../../ReactNative/requireNativeComponent');

var PickerIOS = createReactClass({
  displayName: 'PickerIOS',
  mixins: [NativeMethodsMixin],

  propTypes: {
    ...ViewPropTypes,
    itemStyle: itemStylePropType,
    onValueChange: PropTypes.func,
    selectedValue: PropTypes.any, // string or integer basically
  },

  getInitialState: function() {
    return this._stateFromProps(this.props);
  },

  componentWillReceiveProps: function(nextProps) {
    this.setState(this._stateFromProps(nextProps));
  },

  // Translate PickerIOS prop and children into stuff that RCTPickerIOS understands.
  _stateFromProps: function(props) {
    var selectedIndex = 0;
    var items = [];
    React.Children.toArray(props.children).forEach(function (child, index) {
      if (child.props.value === props.selectedValue) {
        selectedIndex = index;
      }
      items.push({
        value: child.props.value,
        label: child.props.label,
        textColor: processColor(child.props.color),
      });
    });
    return {selectedIndex, items};
  },

  render: function() {
    return (
      <View style={this.props.style}>
        <RCTPickerIOS
          ref={picker => this._picker = picker}
          style={[styles.pickerIOS, this.props.itemStyle]}
          items={this.state.items}
          selectedIndex={this.state.selectedIndex}
          onChange={this._onChange}
          onStartShouldSetResponder={() => true}
          onResponderTerminationRequest={() => false}
        />
      </View>
    );
  },

  _onChange: function(event) {
    if (this.props.onChange) {
      this.props.onChange(event);
    }
    if (this.props.onValueChange) {
      this.props.onValueChange(event.nativeEvent.newValue, event.nativeEvent.newIndex);
    }

    // The picker is a controlled component. This means we expect the
    // on*Change handlers to be in charge of updating our
    // `selectedValue` prop. That way they can also
    // disallow/undo/mutate the selection of certain values. In other
    // words, the embedder of this component should be the source of
    // truth, not the native component.
    if (this._picker && this.state.selectedIndex !== event.nativeEvent.newIndex) {
      this._picker.setNativeProps({
        selectedIndex: this.state.selectedIndex
      });
    }
  },
});

PickerIOS.Item = class extends React.Component {
  static propTypes = {
    value: PropTypes.any, // string or integer basically
    label: PropTypes.string,
    color: PropTypes.string,
  };

  render() {
    // These items don't get rendered directly.
    return null;
  }
};

var styles = StyleSheet.create({
  pickerIOS: {
    // The picker will conform to whatever width is given, but we do
    // have to set the component's height explicitly on the
    // surrounding view to ensure it gets rendered.
    height: 216,
  },
});

var RCTPickerIOS = requireNativeComponent('RCTPicker', {
  propTypes: {
    style: itemStylePropType,
  },
}, {
  nativeOnly: {
    items: true,
    onChange: true,
    selectedIndex: true,
  },
});

module.exports = PickerIOS;
