// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the Apache 2.0 License.
// See the LICENSE file in the project root for more information.

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Reflection;
using System.Runtime.CompilerServices;

using Microsoft.Scripting;
using Microsoft.Scripting.Ast;
using Microsoft.Scripting.Runtime;
using Microsoft.Scripting.Utils;

using IronPython.Runtime.Types;

namespace IronPython.Runtime {
    public sealed partial class PythonFunction : PythonTypeSlot {
        private static Dictionary<FunctionCallerKey/*!*/, FunctionCaller/*!*/>/*!*/ _functionCallers = new Dictionary<FunctionCallerKey, FunctionCaller>();

        Binding.FastBindResult<T> Binding.IFastInvokable.MakeInvokeBinding<T>(CallSite<T> site, Binding.PythonInvokeBinder binder, CodeContext state, object[] args) {
            if (CanOptimizeCall(binder, args)) {
                int funcCompat = FunctionCompatibility;
                // normal positional argument call w/ all arguments present.

                ParameterInfo[] prms = typeof(T).GetMethod("Invoke").GetParameters();
                Type[] allParams = ArrayUtils.ConvertAll(prms, (inp) => inp.ParameterType);
                if (allParams[2] != typeof(object)) {
                    // no optimized form if we're a CallSite<Site, CodeContext, PythonFunction, ...>
                    return new Binding.FastBindResult<T>();
                }
                allParams = ArrayUtils.Append(allParams, typeof(object));

                Type[] typeParams = new Type[prms.Length - 3];  // skip site, context, func
                for (int i = 3; i < prms.Length; i++) {
                    typeParams[i - 3] = prms[i].ParameterType;
                }
                MethodInfo mi;
                Type funcType, callerType;
                FunctionCaller fc;
                string baseName = "";

                if (args.Length != NormalArgumentCount) {
                    baseName = "Default" + (NormalArgumentCount - args.Length);
                }

                PerfTrack.NoteEvent(PerfTrack.Categories.BindingFast, "PythonFunction");

                switch (args.Length) {
                    case 0:
                        if (!String.IsNullOrEmpty(baseName)) {
                            fc = new FunctionCaller(funcCompat);
                            mi = typeof(FunctionCaller).GetMethod(baseName + "Call0");
                            Debug.Assert(mi != null);
                            return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(typeof(Func<CallSite, CodeContext, object, object>), fc), true);
                        }

                        return new Binding.FastBindResult<T>((T)(object)new Func<CallSite, CodeContext, object, object>(new FunctionCaller(funcCompat).Call0), true);

                    #region Generated Python Function Caller Switch

                    // *** BEGIN GENERATED CODE ***
                    // generated by function: function_caller_switch from: generate_calls.py

                    case 1:                        
                        callerType = typeof(FunctionCaller<>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call1");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 2:                        
                        callerType = typeof(FunctionCaller<,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call2");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 3:                        
                        callerType = typeof(FunctionCaller<,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call3");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 4:                        
                        callerType = typeof(FunctionCaller<,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call4");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 5:                        
                        callerType = typeof(FunctionCaller<,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call5");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 6:                        
                        callerType = typeof(FunctionCaller<,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call6");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 7:                        
                        callerType = typeof(FunctionCaller<,,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call7");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 8:                        
                        callerType = typeof(FunctionCaller<,,,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call8");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 9:                        
                        callerType = typeof(FunctionCaller<,,,,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call9");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 10:                        
                        callerType = typeof(FunctionCaller<,,,,,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call10");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 11:                        
                        callerType = typeof(FunctionCaller<,,,,,,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call11");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 12:                        
                        callerType = typeof(FunctionCaller<,,,,,,,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call12");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);
                    case 13:                        
                        callerType = typeof(FunctionCaller<,,,,,,,,,,,,>).MakeGenericType(typeParams);
                        mi = callerType.GetMethod(baseName + "Call13");
                        Debug.Assert(mi != null);
                        fc = GetFunctionCaller(callerType, funcCompat);
                        funcType = typeof(Func<,,,,,,,,,,,,,,,,>).MakeGenericType(allParams);

                        return new Binding.FastBindResult<T>((T)(object)mi.CreateDelegate(funcType, fc), true);

                    // *** END GENERATED CODE ***

                    #endregion
                }
            }

            PerfTrack.NoteEvent(PerfTrack.Categories.BindingSlow, "PythonFunction " + FunctionCompatibility + " - " + ((Flags & FunctionAttributes.Generator) != 0) + ", " + binder.Signature);
            return new Binding.FastBindResult<T>();
        }

        private bool CanOptimizeCall(Binding.PythonInvokeBinder binder, object[] args) {
            return args.Length >= (NormalArgumentCount - _defaults.Length) &&
                args.Length <= NormalArgumentCount &&
                KeywordOnlyArgumentCount == 0 && // don't optimize if we have keyword-only arguments
                ArgNames.Length < FunctionCallerProperties.MaxGeneratedFunctionArgs &&
                !binder.Signature.HasDictionaryArgument() &&
                !binder.Signature.HasKeywordArgument() &&
                !binder.Signature.HasListArgument() &&
                (Flags & (FunctionAttributes.ArgumentList | FunctionAttributes.KeywordDictionary)) == 0 &&
                !binder.SupportsLightThrow();
        }

        private static FunctionCaller/*!*/ GetFunctionCaller(Type/*!*/ callerType, int funcCompat) {
            FunctionCaller res;
            lock (_functionCallers) {
                var key = new FunctionCallerKey(callerType, funcCompat);
                if (!_functionCallers.TryGetValue(key, out res)) {
                    _functionCallers[key] = res = (FunctionCaller)Activator.CreateInstance(callerType, funcCompat);
                }
            }
            return res;
        }

        class FunctionCallerKey : IEquatable<FunctionCallerKey> {
            public readonly Type CallerType;
            public readonly int FunctionCompat;

            public FunctionCallerKey(Type callerType, int compat) {
                CallerType = callerType;
                FunctionCompat = compat;
            }

            #region IEquatable<FunctionCallerKey> Members

            public bool Equals(FunctionCallerKey other) {
                return CallerType == other.CallerType &&
                    FunctionCompat == other.FunctionCompat;
            }

            #endregion

            public override int GetHashCode() {
                return CallerType.GetHashCode() ^ FunctionCompat;
            }

            public override bool Equals(object obj) {
                FunctionCallerKey key = obj as FunctionCallerKey;
                if (key != null) {
                    return Equals(key);
                }
                return false;
            }
        }
    }

    public class FunctionCaller {
        protected readonly int _compat;

        internal FunctionCaller(int compat) {
            _compat = compat;
        }

        public object Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object>)pyfunc.__code__.Target)(pyfunc);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        #region Generated Python Zero Arg Function Callers

        // *** BEGIN GENERATED CODE ***
        // generated by function: function_callers_0 from: generate_calls.py


        public object Default1Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default2Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default3Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default4Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default5Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default6Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default7Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default8Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default9Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default10Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default11Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9], pyfunc.Defaults[defaultIndex + 10]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default12Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9], pyfunc.Defaults[defaultIndex + 10], pyfunc.Defaults[defaultIndex + 11]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        public object Default13Call0(CallSite site, CodeContext context, object func) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9], pyfunc.Defaults[defaultIndex + 10], pyfunc.Defaults[defaultIndex + 11], pyfunc.Defaults[defaultIndex + 12]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, object>>)site).Update(site, context, func);
        }

        // *** END GENERATED CODE ***

        #endregion
    }

    #region Generated Python Function Callers

    // *** BEGIN GENERATED CODE ***
    // generated by function: function_callers from: generate_calls.py

    class FunctionCallerProperties {
        internal const int MaxGeneratedFunctionArgs = 14;
    }


    public sealed class FunctionCaller<T0> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object>)pyfunc.__code__.Target)(pyfunc, arg0);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default1Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default2Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default3Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default4Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default5Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default6Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default7Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default8Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default9Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default10Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default11Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9], pyfunc.Defaults[defaultIndex + 10]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }

        public object Default12Call1(CallSite site, CodeContext context, object func, T0 arg0) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 1;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9], pyfunc.Defaults[defaultIndex + 10], pyfunc.Defaults[defaultIndex + 11]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, object>>)site).Update(site, context, func, arg0);
        }
    }

    public sealed class FunctionCaller<T0, T1> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default1Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default2Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default3Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default4Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default5Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default6Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default7Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default8Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default9Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default10Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }

        public object Default11Call2(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 2;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9], pyfunc.Defaults[defaultIndex + 10]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, object>>)site).Update(site, context, func, arg0, arg1);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default1Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default2Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default3Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default4Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default5Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default6Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default7Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default8Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default9Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }

        public object Default10Call3(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 3;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8], pyfunc.Defaults[defaultIndex + 9]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, object>>)site).Update(site, context, func, arg0, arg1, arg2);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default1Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default2Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default3Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default4Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default5Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default6Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default7Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default8Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }

        public object Default9Call4(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 4;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7], pyfunc.Defaults[defaultIndex + 8]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default1Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default2Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default3Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default4Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default5Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default6Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default7Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }

        public object Default8Call5(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 5;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6], pyfunc.Defaults[defaultIndex + 7]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }

        public object Default1Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 6;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }

        public object Default2Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 6;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }

        public object Default3Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 6;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }

        public object Default4Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 6;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }

        public object Default5Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 6;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }

        public object Default6Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 6;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }

        public object Default7Call6(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 6;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5], pyfunc.Defaults[defaultIndex + 6]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5, T6> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call7(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
        }

        public object Default1Call7(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 7;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
        }

        public object Default2Call7(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 7;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
        }

        public object Default3Call7(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 7;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
        }

        public object Default4Call7(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 7;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
        }

        public object Default5Call7(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 7;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
        }

        public object Default6Call7(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 7;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4], pyfunc.Defaults[defaultIndex + 5]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5, T6, T7> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call8(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7);
        }

        public object Default1Call8(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 8;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7);
        }

        public object Default2Call8(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 8;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7);
        }

        public object Default3Call8(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 8;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7);
        }

        public object Default4Call8(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 8;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7);
        }

        public object Default5Call8(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 8;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3], pyfunc.Defaults[defaultIndex + 4]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5, T6, T7, T8> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call9(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
        }

        public object Default1Call9(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 9;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
        }

        public object Default2Call9(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 9;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
        }

        public object Default3Call9(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 9;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
        }

        public object Default4Call9(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 9;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2], pyfunc.Defaults[defaultIndex + 3]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call10(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
        }

        public object Default1Call10(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 10;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
        }

        public object Default2Call10(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 10;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
        }

        public object Default3Call10(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 10;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1], pyfunc.Defaults[defaultIndex + 2]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call11(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9, T10 arg10) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10);
        }

        public object Default1Call11(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9, T10 arg10) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 11;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10);
        }

        public object Default2Call11(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9, T10 arg10) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 11;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, pyfunc.Defaults[defaultIndex + 0], pyfunc.Defaults[defaultIndex + 1]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call12(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9, T10 arg10, T11 arg11) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11);
        }

        public object Default1Call12(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9, T10 arg10, T11 arg11) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                int defaultIndex = pyfunc.Defaults.Length - pyfunc.NormalArgumentCount + 12;
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, pyfunc.Defaults[defaultIndex + 0]);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11);
        }
    }

    public sealed class FunctionCaller<T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12> : FunctionCaller {
        public FunctionCaller(int compat) : base(compat) { }

        public object Call13(CallSite site, CodeContext context, object func, T0 arg0, T1 arg1, T2 arg2, T3 arg3, T4 arg4, T5 arg5, T6 arg6, T7 arg7, T8 arg8, T9 arg9, T10 arg10, T11 arg11, T12 arg12) {
            if (func is PythonFunction pyfunc && pyfunc.FunctionCompatibility == _compat) {
                return ((Func<PythonFunction, object, object, object, object, object, object, object, object, object, object, object, object, object, object>)pyfunc.__code__.Target)(pyfunc, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12);
            }

            return ((CallSite<Func<CallSite, CodeContext, object, T0, T1, T2, T3, T4, T5, T6, T7, T8, T9, T10, T11, T12, object>>)site).Update(site, context, func, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12);
        }
    }

    // *** END GENERATED CODE ***

    #endregion        
}
