// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the Apache 2.0 License.
// See the LICENSE file in the project root for more information.

using MSAst = System.Linq.Expressions;

using AstUtils = Microsoft.Scripting.Ast.Utils;

namespace IronPython.Compiler.Ast {
    using Ast = MSAst.Expression;

    public class ExpressionStatement : Statement {
        private readonly Expression _expression;

        public ExpressionStatement(Expression expression) {
            _expression = expression;
        }

        public Expression Expression {
            get { return _expression; }
        }

        public override MSAst.Expression Reduce() {
            MSAst.Expression expression = _expression;

            return ReduceWorker(expression);
        }

        private MSAst.Expression ReduceWorker(MSAst.Expression expression) {
            if (Parent.PrintExpressions) {
                expression = Ast.Call(
                    AstMethods.PrintExpressionValue,
                    Parent.LocalContext,
                    ConvertIfNeeded(expression, typeof(object))
                );
            }

            return GlobalParent.AddDebugInfoAndVoid(expression, _expression.Span);
        }

        public override void Walk(PythonWalker walker) {
            if (walker.Walk(this)) {
                _expression?.Walk(walker);
            }
            walker.PostWalk(this);
        }

        public override string Documentation {
            get {
                if (_expression is ConstantExpression ce) {
                    return ce.Value as string;
                }
                return null;
            }
        }

        internal override bool CanThrow {
            get {
                return _expression.CanThrow;
            }
        }
    }
}
