#pragma once

#define CPPHTTPLIB_OPENSSL_SUPPORT

#include <string>
#include <iostream>
#include <math.h>

#include "external/httplib.h"

#include "external/json.hpp"

#include <opencv2/imgcodecs.hpp>

using namespace std;
using json = nlohmann::json;

// base64 functions - start
static const std::string base64_chars =
"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
"abcdefghijklmnopqrstuvwxyz"
"0123456789+/";


static inline bool is_base64(unsigned char c) {
    return (isalnum(c) || (c == '+') || (c == '/'));
}

std::string base64_encode(unsigned char const* bytes_to_encode, unsigned int in_len)
{
    std::string ret;
    int i = 0;
    int j = 0;
    unsigned char char_array_3[3];
    unsigned char char_array_4[4];

    while (in_len--) {
        char_array_3[i++] = *(bytes_to_encode++);
        if (i == 3) {
            char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
            char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
            char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
            char_array_4[3] = char_array_3[2] & 0x3f;
            for (i = 0; (i < 4); i++)
                ret += base64_chars[char_array_4[i]];
            i = 0;
        }
    }

    if (i)
    {
        for (j = i; j < 3; j++)
            char_array_3[j] = '\0';
        char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
        char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
        char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);

        for (j = 0; (j < i + 1); j++)
            ret += base64_chars[char_array_4[j]];
        while ((i++ < 3))
            ret += '=';

    }
    return ret;

}

std::string base64_decode(std::string const& encoded_string) {
    size_t in_len = encoded_string.size();
    int i = 0;
    int j = 0;
    int in_ = 0;
    unsigned char char_array_4[4], char_array_3[3];
    std::string ret;

    while (in_len-- && (encoded_string[in_] != '=') && is_base64(encoded_string[in_])) {
        char_array_4[i++] = encoded_string[in_]; in_++;
        if (i == 4) {
            for (i = 0; i < 4; i++)
                char_array_4[i] = base64_chars.find(char_array_4[i]) & 0xff;

            char_array_3[0] = (char_array_4[0] << 2) + ((char_array_4[1] & 0x30) >> 4);
            char_array_3[1] = ((char_array_4[1] & 0xf) << 4) + ((char_array_4[2] & 0x3c) >> 2);
            char_array_3[2] = ((char_array_4[2] & 0x3) << 6) + char_array_4[3];

            for (i = 0; (i < 3); i++)
                ret += char_array_3[i];
            i = 0;
        }
    }

    if (i) {
        for (j = 0; j < i; j++)
            char_array_4[j] = base64_chars.find(char_array_4[j]) & 0xff;

        char_array_3[0] = (char_array_4[0] << 2) + ((char_array_4[1] & 0x30) >> 4);
        char_array_3[1] = ((char_array_4[1] & 0xf) << 4) + ((char_array_4[2] & 0x3c) >> 2);

        for (j = 0; (j < i - 1); j++) ret += char_array_3[j];
    }

    return ret;
}

std::string get_base64_from_jpg_(std::string Image_name_path) {
    cv::Mat img = cv::imread(Image_name_path, cv::IMREAD_COLOR);
    std::string encoded{ "" };
    if (!img.empty())
    {
        std::vector<uchar> buf;
        cv::imencode(".jpg", img, buf);
        auto* enc_msg = reinterpret_cast<unsigned char*>(buf.data());
        encoded = base64_encode(enc_msg, buf.size());
        return encoded;
    }
    return encoded;
}

std::string encode(cv::Mat img) {
    std::string encoded{ "" };
    if (!img.empty())
    {
        std::vector<uchar> buf;
        cv::imencode(".jpg", img, buf);
        auto* enc_msg = reinterpret_cast<unsigned char*>(buf.data());
        encoded = base64_encode(enc_msg, buf.size());
        return encoded;
    }
    return encoded;
}

std::string get_base64_from_jpg(std::string Image_name_path) {
    cv::Mat img = cv::imread(Image_name_path, cv::IMREAD_COLOR);
    return encode(img);
}

std::string get_base64_from_jpg(cv::Mat img) {
    return encode(img);
}



void save_jpeg(std::string Image_name_path, std::string encoded_base64)
{
    if (Image_name_path != "" and encoded_base64 != "") {
        std::string dec_jpg = base64_decode(encoded_base64);
        std::vector<uchar> data(dec_jpg.begin(), dec_jpg.end());
        cv::Mat img = cv::imdecode(cv::Mat(data), 1);
        cv::imwrite(Image_name_path, img);
    }
}
// base64 functions - end

class ErrorAPI
{
public:
    int status{ 0 };
    std::string error_code{ "" };
    std::string message{ "" };
    int retry_after;
    std::string http_status_code{ "" };

    std::vector<string> loc{};
    std::string msg{ "" };
    std::string type{ "" };

    std::string cert_error_string{ "" };
};

void APIError(json input, int code) {
    if (!((input.contains("code") && input.contains("message")) || input.contains("detail")))
    {
        return;
    }
    for (auto& items : input.items())
    {
        if (input[items.key()].is_null()) input[items.key()] = "";
    }
    
    ErrorAPI errorAPI;

    errorAPI.status = code;
    if (input.contains("message")) { errorAPI.message = input["message"]; }
    if (input.contains("code")) { errorAPI.error_code = input["code"]; }
    if (input.contains("retry_after")) { errorAPI.retry_after = int(input["retry_after"]); }
    if (input.contains("http_status_code")) { errorAPI.http_status_code = input["http_status_code"]; }
    if (input.contains("detail"))
    {
        json tmp1json = input["detail"].front();
        errorAPI.msg = tmp1json["msg"];
        errorAPI.type = tmp1json["type"];
    }
    throw errorAPI;
}

class SortOrder {
 public:
    static inline std::string ASCENDING = "ASC";
    static inline std::string DESCENDING = "DESC";
};

class OrderBy {
 public:
     static inline std::string NAME = "name";
     static inline std::string DATE_OF_BIRTH = "date_of_birth";
     static inline std::string GENDER = "gender";
     static inline std::string NATIONALITY = "nationality";
     static inline std::string MODIFIED_DATE = "modified_date";
     static inline std::string CREATE_DATE = "create_date";
 };

class Options {
public:
    int skip= 0 ;
    int take = 20 ;
    std::string order= SortOrder::ASCENDING;
    std::string order_by= OrderBy::NAME;
    std::string search = "";
};

class ConnectionError
{
public:
    std::string error_code;
    std::string message;
};

class ErrorValidation
{
    std::vector<string> loc;
    std::string msg;
    std::string type;
};

struct SearchMode {
public:
    inline static const std::string  ACCURATE = "ACCURATE";
    inline static const std::string  FAST = "FAST";
};

class SearchRequest {
public:
    double min_score{0.7};
    std::string search_mode{SearchMode::FAST};
    std::vector<string> images;
    std::string collection_id;
    SearchRequest(std::vector<string> images_, float min_score_, std::string search_mode_,std::string collectionid_="") {
        min_score = min_score_;  images = images_; search_mode = search_mode_; collection_id = collectionid_;
    }
    SearchRequest()
    {
    }
};

struct PersonOrderBy {
public:
    inline static const std::string NAME = "name";
    inline static const std::string DATE_OF_BIRTH = "date_of_birth";
    inline static const std::string GENDER = "gender";
    inline static const std::string NATIONALITY = "nationality";
    inline static const std::string MODIFIED_DATE = "modified_date";
    inline static const std::string CREATE_DATE = "create_date";
};

struct Gender {
public:
    inline static const std::string MALE = "M";
    inline static const std::string FEMALE = "F";
};




struct OS {
public:
    inline static const std::string DESKTOP = "DESKTOP";
    inline static const std::string ANDROID = "ANDROID";
    inline static const std::string IOS = "IOS";
};

class CollectionBase {
public:
    std::string name;
    std::string description;
    std::string id;
    CollectionBase(std::string name_, std::string note_,std::string id_="") {
        name = name_;
        description = note_;
        id = id_ ;
    }
    CollectionBase()
    {

    }
};

class Collection: public CollectionBase
{
public:
    
    int count;
    std::string create_date;
    std::string modified_date;
    Collection(std::string name_, std::string description_, std::string create_date_="", std::string modified_date_ = "", std::string id_ = "") :
        CollectionBase(name_, description_, id_), create_date(create_date_), modified_date(modified_date_) {
        count = 0;
    }
    
};

class CollectionCount {
public:
    int count;
    std::vector<Collection> collection;
};

class PersonBase {
public:

    std::string name;
    std::string email;
    std::string mobile;
    std::string gender;
    std::vector<string> images;
    std::string dob;
    std::string id; 
    std::string nationality;
    std::vector<Collection> collections;
    std::vector<cv::Mat> cv_images;
    std::string notes;
    bool is_bulk_insert;
public: 
    PersonBase()
    {
        id = "854c3e44-913b-4590-a582-fc8abb6a9bcb"; 
        name = "";
        gender = "";
        nationality="";
        dob = "";
        email= "";
        collections = {};
        notes = "";
        mobile = "";
        is_bulk_insert = false;
        cv_images = {};
        images = {};

    }
    PersonBase(std::string name_, std::vector<string> images_) {
        name = name_; images = images_; mobile = ""; email = ""; gender = "";
        dob = ""; nationality = ""; collections = {}; notes = ""; is_bulk_insert = false; id = "854c3e44-913b-4590-a582-fc8abb6a9bcb";
    }
    PersonBase(std::string name_, std::string email_, std::string mobile_,std::string gender_ , std::vector<string> images_
        ,std::string dob_,std::string nationality_,std::vector<Collection> collection_,std::string notes_, bool isBulkInsert, std::string id_ = "854c3e44-913b-4590-a582-fc8abb6a9bcb")
    {
        name = name_; email = email_; mobile = mobile_; images = images_; gender = gender_;
        dob = dob_; nationality = nationality_; collections = collection_; notes = notes_; is_bulk_insert = isBulkInsert; id = id_;
        
    }
    void print() 
    {
        std::vector<string> base64_images{};
        for (auto itr : images) 
        {
            std::string temp = get_base64_from_jpg(itr);
            base64_images.push_back(temp);
        }
    }

};

class Thumbnail
{
public:
    std::string id;
    std::string thumbnail_b64;
    std::string save_jpeg(std::string Image_name_path)
    {
        if (Image_name_path != "") {
            std::string dec_jpg = base64_decode(thumbnail_b64);
            std::vector<uchar> data(dec_jpg.begin(), dec_jpg.end());
            cv::Mat img = cv::imdecode(cv::Mat(data), 1);
            cv::imwrite(Image_name_path, img);
            return Image_name_path;
        }
    }
};

class Person
{
private:
    std::string BACKEND;
    std::string api_key;
    int port;
public:
    std::string id;
    std::string name;
    std::string gender;
    std::string date_of_birth;
    std::string nationality;
    std::string create_date;
    std::string modified_date;
    std::vector<string> images;
    std::vector<cv::Mat> cv_images;
    std::vector<Collection> collections;
    std::string note;
    std::string email;
    std::vector<Thumbnail> thumbnails;
    bool is_bulk_insert;
public:
    void setter(std::string backend_, std::string key_,  int port_) {
        api_key = key_;
        BACKEND = backend_;
        port = port_;
    }
    Person()
    {

    }
    Person(std::string id_, std::string name_="", std::string gender_ = "", std::string date_of_birth_ = "",
        std::string nationality_="", std::vector<Collection> collections_ = {},std::vector<string> images_={}, std::string note_ = "", bool is_bulk_insert_ = false,std::vector<Thumbnail> thumbnails_={},std::string email_="")
    {
        id = id_, name = name_; gender = gender_; date_of_birth = date_of_birth_;
        nationality = nationality_; collections = collections_; note = note_; is_bulk_insert = is_bulk_insert_;
    }
    void get_image()
    {

    }
};

class PersonCount {
public:
    int count;
    std::vector<Person> persons;
};

class PersonSearchResult 
{
private:
    std::string BACKEND;
    std::string api_key;
    int port;
public:
    std::string id;
    std::string name;
    std::string gender;
    std::string date_of_birth;
    std::string nationality;
    std::string create_date;
    std::string modified_date;
    double score;
    std::vector<string> images;
    std::vector<Collection> collections;
    std::string note;
    std::vector<Thumbnail> thumbnails;
    bool is_bulk_insert;
public:
    void setter(std::string backend_, std::string key_, int port_) {
        api_key = key_;
        BACKEND = backend_;
        port = port_;
    }
    PersonSearchResult(std::string id_="", std::string name_ = "", std::string gender_ = "", std::string date_of_birth_ = "",
        std::string nationality_ = "", std::vector<Collection> collections_ = {}, std::vector<string> images_ = {}, std::string note_ = "", bool is_bulk_insert_ = false, std::vector<Thumbnail> thumbnails = {},double score_=0)
    {
        id = id_, name = name_; gender = gender_; date_of_birth = date_of_birth_; score = score_;
        nationality = nationality_; collections = collections_; note = note_; is_bulk_insert = is_bulk_insert_;
    }
};

std::vector<PersonSearchResult> JsonToPerson2(json json_obj)
{
    std::vector<PersonSearchResult> return_Persons;
    std::vector<Thumbnail> return_Thumbnail;
    std::vector<Collection> return_Collection;
    for (auto& item : json_obj.items())
    {
        // fill null elements in JSON with EmptyString
        for (auto& it : item.value()) {
            if (it.is_null()) {
                it = "";
            }
        }
        //cout << item.value() << "\n\n\n";  

        PersonSearchResult aPerson;
        std::vector<Thumbnail> return_Thumbnail;
        std::vector<Collection> return_Collection;
        aPerson.id = item.value()["id"].get<std::string>();
        aPerson.name = item.value()["name"].get<std::string>();

        aPerson.date_of_birth = item.value()["date_of_birth"].get<std::string>();
        aPerson.gender = item.value()["gender"].get<std::string>();
        aPerson.nationality = item.value()["nationality"].get<std::string>();
        aPerson.create_date = item.value()["create_date"].get<std::string>();
        aPerson.modified_date = item.value()["modified_date"].get<std::string>();
        aPerson.score = double(item.value()["score"]);


        for (auto& temp_thumbnail : item.value()["thumbnails"])
        {
            Thumbnail aThumbnail;
            aThumbnail.id = temp_thumbnail["id"].get<std::string>();
            aThumbnail.thumbnail_b64 = temp_thumbnail["thumbnail"].get<std::string>();
            return_Thumbnail.push_back(aThumbnail);
        }
        aPerson.thumbnails = return_Thumbnail;
        for (auto& temp_collection : item.value()["collections"])
        {
            Collection aCollection{ "","" };
            aCollection.id = temp_collection["id"].get<std::string>();
            aCollection.name = temp_collection["name"].get<std::string>();
            aCollection.description = temp_collection["description"].get<std::string>();
            aCollection.create_date = temp_collection["create_date"].get<std::string>();
            aCollection.modified_date = temp_collection["modified_date"].get<std::string>();
            aCollection.count = int(temp_collection["count"]);
            return_Collection.push_back(aCollection);
        }
        aPerson.collections = return_Collection;
        return_Persons.push_back(aPerson);
    }
    return return_Persons;

}

class DetectRequest
{
public:
    std::string collection_id;
    double min_score;
    std::string search_mode;
    DetectRequest(std::string collection_id_, double min_score_, std::string search_mode_)
    {
        collection_id = collection_id_; min_score = min_score_; search_mode = search_mode_;
    }

};

class VerificationRequest
{
public:
    std::string id;
    double min_score;
    std::vector<string> images;
    std::string search_mode;
    VerificationRequest(std::string id_, std::vector<string> images_, double min_score_, std::string search_mode_) {
        id = id_; min_score = min_score_; images = images_; search_mode = search_mode_;
    }
};

class SearchModeClass
{
public:
    static const  std::string   FAST;
    static const  std::string   ACCURATE;
    std::string FAST1 =  "FAST" ;
    std::string ACCURATE1 =  "ACCURATE" ;
};

class Box
{public:
    int left=0, top=0, bottom=0, right = 0;
};

class LandMarks
{
public:
    int left_eye[2];
    int right_eye[2];
    int nose[2];
    int left_mouth[2];
    int right_mouth[2];
};

class DetectionResponseItem
{
public:
    Box theBox;
    LandMarks theLandMarks;
    double detection_score;
    std::string thumbnail;
    std::vector<PersonSearchResult> persons;
};





class CompareRequest 
{
public:
    std::vector<string> gallery;
    std::vector<string> probe;
    std::string mode;
};

class LivenessRequest {
public:
    std::string os;
    std::string image;
};

class LivenessResponse
{
public:
    double score{ 0 };
};

// API classes

class Compare
{
private:
    int port;
    std::string BACKEND_URL;
    std::string DEVELOPER_KEY;
public:
    void setter(std::string BACKEND_URL_, std::string DEVELOPER_KEY_, int port_)
    {
        BACKEND_URL = BACKEND_URL_;
        DEVELOPER_KEY = DEVELOPER_KEY_;
        port = port_;
    }
    double compare_image_sets(CompareRequest CR)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/compare";
        std::vector<char> route_verify(str.begin(), str.end());
        route_verify.push_back('\0');
        
        const char* route_{ &route_verify[0] };
        json body;
        for (auto& pic : CR.gallery) {
            body["gallery"].push_back(get_base64_from_jpg(pic));
        }
        for (auto& pro : CR.probe) {
            body["probe"].push_back(get_base64_from_jpg(pro));
        }
        std::string json_str = body.dump();
        double score=0;
        if (auto res = cli.Post(route_, headers, json_str, "application/json"))
        {
            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));
            score = returned_JSON["score"];
        }
        else {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return score;
    }
};

class SearchCropRequest
{
public:
    std::string collection_id;
    std::vector<std::string> images;
    double min_score;
    std::string search_mode;
};

class Persons
{
private:
    int port;
    std::string BACKEND_URL;
    std::string DEVELOPER_KEY;
    Person createAPerson(json returned_json)
    {
        std::vector<Thumbnail> return_thumbnail{};
        Person aPerson{ "" };
        std::vector<Collection> return_collection{};

        for (auto& items : returned_json.items())
        {
            if (returned_json[items.key()].is_null()) returned_json[items.key()] = "";
        }

        if (returned_json.find("name") != returned_json.end()) {
            aPerson.name = (returned_json["name"].get<std::string>());
            aPerson.date_of_birth = (returned_json["date_of_birth"].get<std::string>());
            aPerson.gender = (returned_json["gender"].get<std::string>());
            aPerson.nationality = (returned_json["nationality"].get<std::string>());
            aPerson.id = (returned_json["id"].get<std::string>());
            aPerson.note = (returned_json["notes"].get<std::string>());
            for (auto& resul : returned_json["thumbnails"].items()) {
                Thumbnail aThumbnail;
                aThumbnail.id = resul.value()["id"].get<std::string>();
                aThumbnail.thumbnail_b64 = resul.value()["thumbnail"].get<std::string>();
                return_thumbnail.push_back(aThumbnail);
            }
            for (auto& resul : returned_json["collections"].items()) {
                Collection temp{ "","" };
                temp.id = resul.value()["id"];
                temp.description = resul.value()["description"];
                temp.count = resul.value()["count"];
                temp.create_date = resul.value()["create_date"];
                temp.modified_date = resul.value()["modified_date"];
                temp.name = resul.value()["name"];
                return_collection.push_back(temp);
            }
            aPerson.setter(BACKEND_URL, DEVELOPER_KEY, port);
            aPerson.collections = return_collection;
            aPerson.thumbnails = return_thumbnail;
        }

        return aPerson;


    }

    json personToJson(Person inPerson)
    {

        json test;
        json j_body;
        for (auto itr : inPerson.collections) {
            std::string temp = itr.id;
            j_body["collections"].push_back(temp);
        }
        if (inPerson.name.size() != 0)
            j_body["name"] = inPerson.name;
        if (inPerson.gender.size() != 0)
            j_body["gender"] = inPerson.gender;
        j_body["nationality"] = inPerson.nationality;
        j_body["email"] = inPerson.email;
        j_body["notes"] = inPerson.note;
        j_body["id"] = inPerson.id;
        if (inPerson.date_of_birth.size() != 0) {
            j_body["date_of_birth"] = inPerson.date_of_birth;
        }
        //else j_body["date_of_birth"] = "";
        j_body["is_bulk_insert"] = inPerson.is_bulk_insert;

        for (auto itr : inPerson.images)
        {
            std::string images_base64_temp = get_base64_from_jpg(itr);
            j_body["images"].push_back(images_base64_temp);

        }
        for (auto itr : inPerson.cv_images)
        {
            j_body["images"].push_back(encode(itr));
        }
        return j_body;
    }

    Person JsonToOnePerson(json json_obj)
    {
        Person return_Person{ "" };
        return_Person.id = json_obj["id"];
        return_Person.create_date = json_obj["create_date"];
        return_Person.date_of_birth = json_obj["create_date"];
        return_Person.email = json_obj["create_date"];
        return_Person.gender = json_obj["create_date"];
        return_Person.is_bulk_insert = json_obj["create_date"];

        return return_Person;

    }
public:
    void setter(std::string BACKEND_URL_, std::string DEVELOPER_KEY_,int port_)
    {
        BACKEND_URL = BACKEND_URL_;
        DEVELOPER_KEY = DEVELOPER_KEY_;
        port = port_;
    }
    
    std::string delete_person(std::string id_)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };

        std::string str="/person/"+id_;
        std::vector<char> route_customer(str.begin(), str.end());
        std::string id_return;
        route_customer.push_back('\0');

        const char* route_{ &route_customer[0]};
        if (auto res = cli.Delete(route_, headers))
        {
            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));
            if (returned_JSON.contains("id"))
                id_return = returned_JSON["id"];
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return id_return;
    }

    

    std::vector<Person> JsonToPerson(json json_obj)
    {
        std::vector<Person> return_Persons;
        std::vector<Thumbnail> return_Thumbnail;
        std::vector<Collection> return_Collection;
        for (auto& item : json_obj["persons"].items())
        {

            // fill null elements in JSON with EmptyString
            for (auto& it : item.value()) {
                if (it.is_null()) {
                    it = "";
                }
            }

            Person aPerson{ "" };
            std::vector<Thumbnail> return_Thumbnail;
            std::vector<Collection> return_Collection;
            aPerson.id = item.value()["id"].get<std::string>();
            aPerson.name = item.value()["name"].get<std::string>();

            aPerson.date_of_birth = item.value()["date_of_birth"].get<std::string>();
            aPerson.gender = item.value()["gender"].get<std::string>();
            aPerson.nationality = item.value()["nationality"].get<std::string>();
            aPerson.create_date = item.value()["create_date"].get<std::string>();
            aPerson.modified_date = item.value()["modified_date"].get<std::string>();


            for (auto& temp_thumbnail : item.value()["thumbnails"])
            {
                Thumbnail aThumbnail;
                aThumbnail.id = temp_thumbnail["id"].get<std::string>();
                aThumbnail.thumbnail_b64 = temp_thumbnail["thumbnail"].get<std::string>();
                return_Thumbnail.push_back(aThumbnail);
            }
            aPerson.thumbnails = return_Thumbnail;
            for (auto& temp_collection : item.value()["collections"])
            {
                Collection aCollection{ "","" };
                aCollection.id = temp_collection["id"].get<std::string>();
                aCollection.name = temp_collection["name"].get<std::string>();
                aCollection.description = temp_collection["description"].get<std::string>();
                aCollection.create_date = temp_collection["create_date"].get<std::string>();
                aCollection.modified_date = temp_collection["modified_date"].get<std::string>();
                aCollection.count = int(temp_collection["count"]);
                return_Collection.push_back(aCollection);
            }
            aPerson.collections = return_Collection;
            return_Persons.push_back(aPerson);

        }
        return return_Persons;

    }

    std::vector<Person> getPersonInCollection(Collection collection_,Options ops) 
    {
        std::vector<Person> return_Persons;
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };

        std::string options_string = "?skip=" +std::to_string(ops.skip)+"&take="+std::to_string(ops.take)
            +"&order="+ops.order+"&order_by="+ops.order_by;

        std::string temp_id = collection_.id;
        std::string str = "/persons/" + temp_id + options_string;
        std::vector<char> route_customer(str.begin(), str.end());
        route_customer.push_back('\0');
        const char* route_{ &route_customer[0] };
        if (auto res = cli.Get(route_, headers))
        {
            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));
            return_Persons =JsonToPerson(returned_JSON);
            return return_Persons;
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }

    }

    PersonCount list(Options ops)
    {

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/persons?skip=" + std::to_string(ops.skip) + "&take=" + std::to_string(ops.take);
        if (ops.order != "") str += "&order=" + ops.order;
        if (ops.order_by != "")    str += "&order_by=" + ops.order_by;
        if (ops.search != "") str += "&search=" + ops.search;
        //cout << "====  " << str << "   =====\n";
        std::vector<char> route_str(str.begin(), str.end());
        route_str.push_back('\0');
        PersonCount personcount;
        const char* route_{ &route_str[0] };
        if (auto res = cli.Get(route_, headers))
        {
            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));
            personcount.count = int(returned_JSON["count"]);

            json persons_JSON = returned_JSON["persons"];
            std::vector<Person> PersonList = JsonToPerson(returned_JSON);
            personcount.persons = PersonList;
            return personcount;
        }
    }

    Person get(std::string person_id)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };

        std::string str = "/person/"+person_id;
        std::vector<char> route_person_get(str.begin(), str.end());
        route_person_get.push_back('\0');
        const char* route_{ &route_person_get[0] };
        Person return_person{""};
        if (auto res = cli.Get(route_, headers))
        {
            //cout <<"status: "<< res->status << "\n";
            json returned_json = json::parse(res->body);
            APIError(returned_json, int(res->status));
            
            // return if no exception
            return_person = createAPerson(returned_json);
        }
        else
        {
            //ConnectionError(cli);
            //cout << "error code: " << res.error() << std::endl;
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";
            
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return return_person;
    }
    
    Person update(Person personToUpdate) 
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        Person aPerson{ "" };
        PersonSearchResult aaPerson;
        json jsonTosend=personToJson(personToUpdate);
        for (auto& items : jsonTosend.items())
        {
            //cout << jsonTosend[items.key()]<<" "<< "\n"; //jsonTosend[items.value()] <<
            if (jsonTosend[items.key()].is_null()) {
                jsonTosend[items.key()] = "";
            }
        }
        std::string jsonBody = jsonTosend.dump();
        std::string str = "/person";
        std::vector<char> route_collection_get(str.begin(), str.end());
        route_collection_get.push_back('\0');
        const char* route_{ &route_collection_get[0] };
        if (auto res = cli.Patch(route_, headers, jsonBody, "application/json"))
        {
            json returned_JSON = json::parse(res->body);
            if (int(res->status) != 200)
                APIError(returned_JSON, int(res->status));
            aPerson = createAPerson(returned_JSON);
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }

        return aPerson;
    }

    PersonCount get(Options& ops)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/persons?skip=" + std::to_string(ops.skip) + "&take=" + std::to_string(ops.take);
        if (ops.order != "") str += "&order=" + ops.order;
        if (ops.order_by != "")    str+= "&order_by=" + ops.order_by;
        if (ops.search !="") str+= "&search=" + ops.search;
        //cout << "====  " << str << "   =====\n";
        std::vector<char> route_str(str.begin(), str.end()); 
        route_str.push_back('\0');
        PersonCount personcount;
        const char* route_{ &route_str[0] };
        if (auto res = cli.Get(route_, headers))
        {
            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));
            personcount.count = int(returned_JSON["count"]);
            
            json persons_JSON= returned_JSON["persons"];
            std::vector<Person> PersonList = JsonToPerson(returned_JSON);
            personcount.persons = PersonList;
            return personcount;
        }
    }

    Person create(PersonBase PB)
    {
        using json = nlohmann::json;
        json j_body;
        for (auto itr : PB.collections) {
            std::string temp = itr.id; 
            j_body["collections"].push_back(temp);
        }

        if (PB.id == "854c3e44-913b-4590-a582-fc8abb6a9bcb")
        { 
        }
        else{
            j_body["id"] = PB.id;
        }
        //if (PB.id != "")  
        //    j_body["id"] = PB.id;
        if (PB.name!= "")
            j_body["name"] = PB.name;
        if (PB.gender!="")
            j_body["gender"] = PB.gender;
        if (PB.nationality!="")
            j_body["nationality"] = PB.nationality;
        if (PB.notes!="")
            j_body["notes"] = PB.notes;
        if (PB.dob!="")
            j_body["date_of_birth"] = PB.dob;  
        
        j_body["is_bulk_insert"] = PB.is_bulk_insert;
        for (auto itr : PB.images) 
        {
            std::string images_base64_temp = get_base64_from_jpg(itr);
            j_body["images"].push_back(images_base64_temp);
        }
        for (auto itr : PB.collections)
        {
            j_body["collections"].push_back(itr.id);
        }
        for (auto itr : PB.cv_images)
        {
            j_body["images"].push_back(encode(itr));
        }
        std::string json_body = j_body.dump();



#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        const char* route_person{ "/person" };
        if (auto res = cli.Post(route_person, headers, json_body, "application/json"))
        {
            json returned_json = json::parse(res->body);
            if (int(res->status)!=200 || int(res->status) != 201)
                {   APIError(returned_json, int(res->status)); }
            //
            for (auto& items : returned_json.items())
            {
                if (returned_json[items.key()].is_null()) returned_json[items.key()] = "";
            }
            //
            std::vector<Thumbnail> return_thumbnail{};
            Person aPerson{""};
            std::vector<Collection> return_collection{};
            
            if (returned_json.find("name") != returned_json.end()) {
                aPerson.name = ( returned_json["name"].get<std::string>());
                aPerson.date_of_birth = (returned_json["date_of_birth"].get<std::string>());
                aPerson.gender = (returned_json["gender"].get<std::string>());
                aPerson.nationality = (returned_json["nationality"].get<std::string>());
                aPerson.id = (returned_json["id"].get<std::string>());
                for (auto& resul : returned_json["thumbnails"].items()) {
                    Thumbnail aThumbnail;
                    aThumbnail.id = resul.value()["id"].get<std::string>();
                    aThumbnail.thumbnail_b64 = resul.value()["thumbnail"].get<std::string>();
                    return_thumbnail.push_back(aThumbnail);
                }
                
                for (auto& resul : returned_json["collections"].items()) {
                    Collection temp{ "","" };
                    temp.id = resul.value()["id"];
                    temp.description = resul.value()["description"];
                    temp.count = resul.value()["count"];
                    temp.create_date = resul.value()["create_date"];
                    temp.modified_date = resul.value()["modified_date"];
                    temp.name = resul.value()["name"];
                    return_collection.push_back(temp);
                    
                }
            }
            aPerson.setter(BACKEND_URL, DEVELOPER_KEY, port);
            aPerson.collections = return_collection;
            aPerson.thumbnails = return_thumbnail;
            return aPerson;
            
        }
        else {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
        }

    }
};


class Search
{
private:
    int port;
    std::string BACKEND_URL;
    std::string DEVELOPER_KEY;
public:
    void setter(std::string BACKEND_URL_, std::string DEVELOPER_KEY_, int port_)
    {
        BACKEND_URL = BACKEND_URL_;
        DEVELOPER_KEY = DEVELOPER_KEY_;
        port = port_;
    }

    std::vector<PersonSearchResult> search_crop(SearchCropRequest SCR)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"accept","application/json"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/search/crops";
        std::vector<char> route_verify(str.begin(), str.end());
        route_verify.push_back('\0');
        const char* route_{ &route_verify[0] };
        json body;
        body["min_score"] = round(SCR.min_score * 100) / 100;
        body["verification_mode"] = SCR.search_mode;
        for (auto itr : SCR.images)
        {
            std::string images_base64_temp = get_base64_from_jpg(itr);
            body["images"].push_back(images_base64_temp);
        }
        std::string json_str = body.dump();
        std::vector<PersonSearchResult> returnPersons;
        if (auto res = cli.Post(route_, headers, json_str, "application/json"))
        {
            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));
            for (json item : returned_JSON.items())
            {
                json aPerson = item.front();
                PersonSearchResult return_Person{ "" };
                return_Person.id = aPerson["id"];
                return_Person.name = aPerson["name"];
                return_Person.create_date = aPerson["create_date"];
                return_Person.date_of_birth = aPerson["date_of_birth"];
                return_Person.nationality = aPerson["nationality"];
                return_Person.gender = aPerson["gender"];
                return_Person.note = aPerson["notes"];
                return_Person.score = aPerson["score"];

                std::vector<Thumbnail> return_Thumbnail;
                std::vector<Collection> return_Collection;

                for (auto& temp_thumbnail : item["thumbnails"])
                {
                    Thumbnail aThumbnail;
                    aThumbnail.id = temp_thumbnail["id"].get<std::string>();
                    aThumbnail.thumbnail_b64 = temp_thumbnail["thumbnail"].get<std::string>();
                    return_Thumbnail.push_back(aThumbnail);
                }
                return_Person.thumbnails = return_Thumbnail;
                for (auto& temp_collection : item["collections"])
                {
                    Collection aCollection{ "","" };
                    aCollection.id = temp_collection["id"].get<std::string>();
                    aCollection.name = temp_collection["name"].get<std::string>();
                    aCollection.description = temp_collection["description"].get<std::string>();
                    aCollection.create_date = temp_collection["create_date"].get<std::string>();
                    aCollection.modified_date = temp_collection["modified_date"].get<std::string>();
                    aCollection.count = int(temp_collection["count"]);
                    return_Collection.push_back(aCollection);
                }
                return_Person.collections = return_Collection;
                returnPersons.push_back(return_Person);
            }
            return returnPersons;
        }
    }

    PersonSearchResult verify(VerificationRequest VRR)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"accept","application/json"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/verify";
        std::vector<char> route_verify(str.begin(), str.end());
        route_verify.push_back('\0');
        const char* route_{ &route_verify[0] };
        json body;
        body["min_score"] = round(VRR.min_score * 100) / 100;
        body["verification_mode"] = VRR.search_mode; // verification_mode ,originally(wrong) is search_mode
        body["id"] = VRR.id;
        for (auto itr : VRR.images)
        {
            std::string images_base64_temp = get_base64_from_jpg(itr);
            body["images"].push_back(images_base64_temp);
        }

        std::string json_str = body.dump();
        //cout << "body is\n" << json_str;;
        std::vector<PersonSearchResult> returnPersons;
        PersonSearchResult returnVerifiedPerson;

        if (auto res = cli.Post(route_, headers, json_str, "application/json"))
        {
            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));
            // return will have match unless APIError.
            json item = returned_JSON["match"];

            // no person match to the input photo
            if (item.is_null()) return returnVerifiedPerson;

            PersonSearchResult aPerson;
            std::vector<Thumbnail> return_Thumbnail;
            std::vector<Collection> return_Collection;

            for (auto& items : item.items())
            {
                if (item[items.key()].is_null()) item[items.key()] = "";
            }

            aPerson.id = item["id"].get<std::string>();
            aPerson.name = item["name"].get<std::string>();
            aPerson.date_of_birth = item["date_of_birth"].get<std::string>();
            aPerson.gender = item["gender"].get<std::string>();
            aPerson.nationality = item["nationality"].get<std::string>();
            aPerson.create_date = item["create_date"].get<std::string>();
            aPerson.modified_date = item["modified_date"].get<std::string>();
            aPerson.score = double(item["score"]);

            for (auto& temp_thumbnail : item["thumbnails"])
            {
                Thumbnail aThumbnail;
                aThumbnail.id = temp_thumbnail["id"].get<std::string>();
                aThumbnail.thumbnail_b64 = temp_thumbnail["thumbnail"].get<std::string>();
                return_Thumbnail.push_back(aThumbnail);
            }
            aPerson.thumbnails = return_Thumbnail;
            for (auto& temp_collection : item["collections"])
            {
                Collection aCollection{ "","" };
                aCollection.id = temp_collection["id"].get<std::string>();
                aCollection.name = temp_collection["name"].get<std::string>();
                aCollection.description = temp_collection["description"].get<std::string>();
                aCollection.create_date = temp_collection["create_date"].get<std::string>();
                aCollection.modified_date = temp_collection["modified_date"].get<std::string>();
                aCollection.count = int(temp_collection["count"]);
                return_Collection.push_back(aCollection);
            }
            aPerson.collections = return_Collection;
            returnVerifiedPerson = aPerson;

        }
        else {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }

        return returnVerifiedPerson;
    }

    std::vector<PersonSearchResult> search(SearchRequest searchRequest_)
    {

        std::vector<PersonSearchResult> returnPersons;

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/search";
        //const char* str1 = "/search";
        std::vector<char> route_customer(str.begin(), str.end());
        route_customer.push_back('\0');

        const char* route_{ &route_customer[0] };
        json body;
        if (searchRequest_.collection_id.length() > 0)
            body["collection_id"] = searchRequest_.collection_id;
        body["min_score"] = round(searchRequest_.min_score * 100) / 100;
        body["search_mode"] = searchRequest_.search_mode;
        for (auto itr : searchRequest_.images)
        {
            std::string images_base64_temp = get_base64_from_jpg(itr);
            body["images"].push_back(images_base64_temp);
        }
        std::string json_str = body.dump();
        if (auto res = cli.Post(route_, headers, json_str, "application/json"))
        {
            json return_persons = json::parse(res->body);
            if (int(res->status) != 200) {
                APIError(return_persons, int(res->status));
            }
            returnPersons = JsonToPerson2(return_persons);
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return returnPersons;
    }

    std::vector<DetectionResponseItem> detect(std::string ImagePath)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };

        const char* route_detect{ "/detect" };
        std::vector<DetectionResponseItem> returnObj;
        json temp;
        temp["image"] = get_base64_from_jpg(ImagePath);
        std::string JsonBody = temp.dump();
        if (auto res = cli.Post(route_detect, headers, JsonBody, "application/json"))
        {
            json returned_JSON = json::parse(res->body);
            if (int(res->status) != 200) {
                APIError(returned_JSON, int(res->status));
            }
            for (auto& item : returned_JSON.items())
            {
                DetectionResponseItem DRI;
                //cout << item.key() <<" AND "<< item.value() << "\n";
                json subitem = item.value();

                Box aBox;
                aBox.bottom = subitem["box"]["bottom"];
                aBox.left = subitem["box"]["left"];
                aBox.right = subitem["box"]["right"];
                aBox.top = subitem["box"]["top"];
                DRI.theBox = aBox;
                DRI.detection_score = subitem["detection_score"];
                LandMarks aLandMark;
                //for (auto& lm: subitem["landmarks"])
                aLandMark.left_eye[0] = subitem["landmarks"]["left_eye"][0];
                aLandMark.left_eye[1] = subitem["landmarks"]["left_eye"][1];
                aLandMark.left_mouth[0] = subitem["landmarks"]["left_mouth"][0];
                aLandMark.left_mouth[1] = subitem["landmarks"]["left_mouth"][1];
                aLandMark.nose[0] = subitem["landmarks"]["nose"][0];
                aLandMark.nose[1] = subitem["landmarks"]["nose"][1];
                aLandMark.right_eye[0] = subitem["landmarks"]["right_eye"][0];
                aLandMark.right_eye[1] = subitem["landmarks"]["right_eye"][1];
                aLandMark.right_mouth[0] = subitem["landmarks"]["right_mouth"][0];
                aLandMark.right_mouth[1] = subitem["landmarks"]["right_mouth"][1];
                DRI.theLandMarks = aLandMark;

                std::string tempThumbnail = subitem["thumbnail"];  // contains thumbnail of person in input photo
                DRI.thumbnail = tempThumbnail;
                std::vector<PersonSearchResult> tempPersons{};
                DRI.persons = tempPersons;
                returnObj.push_back(DRI);
            }
            return returnObj;
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
    }

    std::vector<DetectionResponseItem> detect(std::string ImagePath, DetectRequest DR)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };

        const char* route_detect{ "/detect" };
        std::vector<DetectionResponseItem> returnObj;
        json temp;
        temp["image"] = get_base64_from_jpg(ImagePath);

        temp["search"]["collection_id"] = DR.collection_id;
        temp["search"]["min_score"] = DR.min_score;
        temp["search"]["search_mode"] = DR.search_mode;
        std::string JsonBody = temp.dump();
        if (auto res = cli.Post(route_detect, headers, JsonBody, "application/json"))
        {

            json returned_JSON = json::parse(res->body);
            APIError(returned_JSON, int(res->status));


            for (auto& item : returned_JSON.items())
            {
                DetectionResponseItem DRI;
                //cout << item.key() <<" AND "<< item.value() << "\n";
                json subitem = item.value();

                Box aBox;
                aBox.bottom = subitem["box"]["bottom"];
                aBox.left = subitem["box"]["left"];
                aBox.right = subitem["box"]["right"];
                aBox.top = subitem["box"]["top"];
                DRI.theBox = aBox;

                LandMarks aLandMark;
                //for (auto& lm: subitem["landmarks"])
                aLandMark.left_eye[0] = subitem["landmarks"]["left_eye"][0];
                aLandMark.left_eye[1] = subitem["landmarks"]["left_eye"][1];
                aLandMark.left_mouth[0] = subitem["landmarks"]["left_mouth"][0];
                aLandMark.left_mouth[1] = subitem["landmarks"]["left_mouth"][1];
                aLandMark.nose[0] = subitem["landmarks"]["nose"][0];
                aLandMark.nose[1] = subitem["landmarks"]["nose"][1];
                aLandMark.right_eye[0] = subitem["landmarks"]["right_eye"][0];
                aLandMark.right_eye[1] = subitem["landmarks"]["right_eye"][1];
                aLandMark.right_mouth[0] = subitem["landmarks"]["right_mouth"][0];
                aLandMark.right_mouth[1] = subitem["landmarks"]["right_mouth"][1];
                DRI.theLandMarks = aLandMark;
                DRI.detection_score = subitem["detection_score"];

                std::string tempThumbnail = subitem["thumbnail"];  // contains thumbnail of person in input photo
                DRI.thumbnail = tempThumbnail;
                std::vector<PersonSearchResult> tempPersons;
                for (auto& person : subitem["persons"])
                {
                    PersonSearchResult aPerson;
                    aPerson.name = person["name"];
                    aPerson.id = person["id"];
                    aPerson.score = person["score"];
                    std::vector<Collection> tempCollections;
                    std::vector<Thumbnail> tempThumbnails;

                    for (auto& collection : subitem["collections"]) {
                        Collection aCollection{ "","" };
                        aCollection.count = collection["count"];
                        aCollection.description = collection["description"];
                        aCollection.create_date = collection["create_date"];
                        aCollection.modified_date = collection["modified_date"];
                        aCollection.id = collection["id"];
                        aCollection.name = collection["name"];
                        tempCollections.push_back(aCollection);
                    }
                    aPerson.collections = tempCollections;

                    for (auto& thumbnail : subitem["thumbnails"]) {
                        Thumbnail aThumbnail;
                        aThumbnail.id = thumbnail["id"];
                        aThumbnail.thumbnail_b64 = thumbnail["thumbnail"];
                        tempThumbnails.push_back(aThumbnail);
                    }
                    aPerson.thumbnails = tempThumbnails;

                    tempPersons.push_back(aPerson);
                    DRI.persons = tempPersons;
                }
                returnObj.push_back(DRI);
            }
            // Assembly of return object

        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return returnObj;
    }


};


class Liveness {
private:
    int port;
    std::string BACKEND_URL;
    std::string DEVELOPER_KEY;
public:
    void setter(std::string BACKEND_URL_, std::string DEVELOPER_KEY_, int port_)
    {
        BACKEND_URL = BACKEND_URL_;
        DEVELOPER_KEY = DEVELOPER_KEY_;
        port = port_;
    }
    LivenessResponse check_liveness(LivenessRequest LR)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/liveness";
        std::vector<char> route_liveness(str.begin(), str.end());
        route_liveness.push_back('\0');
        const char* route_{ &route_liveness[0] };

        json JBody;
        LivenessResponse LResponse;
        JBody["os"] = LR.os;
        JBody["image"] = get_base64_from_jpg(LR.image);
        std::string json_body = JBody.dump();
        if (auto res = cli.Post(route_, headers, json_body, "application/json"))
        {
            json result=json::parse(res->body);
            APIError(result, int(res->status));
            LResponse.score = result["liveness_score"];
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return LResponse;
         
    }
};

class Collections {
private:
    int port;
    std::string BACKEND_URL;
    std::string DEVELOPER_KEY;
public:

    void setter(std::string BACKEND_URL_, std::string DEVELOPER_KEY_, int port_) 
    {
        BACKEND_URL = BACKEND_URL_;
        DEVELOPER_KEY = DEVELOPER_KEY_;
        port = port_;
    }
    
    Collection get(std::string collectionID)
    {
        Collection returnCollection{"",""};
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"accept","application/json"},
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"accept","*/*"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/collection/"+collectionID;
        std::vector<char> route_collection_get(str.begin(), str.end());
        route_collection_get.push_back('\0');
        const char* route_{ &route_collection_get[0] };
        //auto res1 = cli.Get(route_, headers);
        if (auto res = cli.Get(route_, headers))
        {
            int error_ = int(res->status);
            json Return_body = json::parse(res->body);
            APIError(Return_body, int(res->status));
            std::string retryAfter;
            if (Return_body.contains("retry_after"))
            {
                retryAfter = Return_body["retry_after"];
            }
            if (Return_body.contains("code") && Return_body.contains("message"))
            {

                ErrorAPI errorAPI;
                errorAPI.status = error_;
                errorAPI.error_code = Return_body["code"];
                errorAPI.message = Return_body["message"];
                throw errorAPI;
            }
            std::string tmp_ = Return_body.dump();
            if (!Return_body.is_null())
            {
                returnCollection.count = int(Return_body["count"]);
                returnCollection.id = Return_body["id"];
                returnCollection.name = Return_body["name"];
                if (!Return_body["description"].is_null() )
                    returnCollection.description = Return_body["description"];
                returnCollection.create_date = Return_body["create_date"];
                returnCollection.modified_date = Return_body["modified_date"];
            }
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";
            
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return returnCollection;

    }

    std::string delectCollection(Collection collection)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/collection/" + collection.id;
        std::vector<char> route_customer(str.begin(), str.end());
        route_customer.push_back('\0');
        const char* route_{ &route_customer[0] };
        std::string deleted_id;
        if (auto res = cli.Delete(route_, headers))
        {
            json returnJson = json::parse(res->body);
            APIError(returnJson, int(res->status));
            if (!returnJson.empty())
                deleted_id = returnJson["id"];
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return deleted_id;
    }

    std::string delete_collection(std::string id_collection_)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif
        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/collection/" + id_collection_;
        std::vector<char> route_customer(str.begin(), str.end());
        route_customer.push_back('\0');
        const char* route_{ &route_customer[0] };
        std::string deleted_id;
        if (auto res = cli.Delete(route_, headers))
        {
            json returnJson = json::parse(res->body);
            if (int(res->status)!=200)
                APIError(returnJson, int(res->status));
            if (!returnJson.empty())
                deleted_id = returnJson["id"];
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return deleted_id;
    }

    CollectionCount gets(Options ops)
    {
        CollectionCount collectioncount;
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/collections?skip=" + std::to_string(ops.skip) + "&take=" + std::to_string(ops.take)
            + "&order=" + ops.order + "&search=" + ops.search;
        //std::string str = "/collections?skip=0&take=20&order=ASC";
        std::vector<char> route_collection_get(str.begin(), str.end());
        route_collection_get.push_back('\0');
        const char* route_{ &route_collection_get[0] };
        std::vector<Collection> results{};
        auto res = cli.Get(route_, headers);
        if (auto res = cli.Get(route_, headers))
        {
            json Return_body = json::parse(res->body);
            APIError(Return_body, int(res->status));
            collectioncount.count = int(Return_body["count"]);
            if (!Return_body.empty()) {
                for (auto& item : Return_body["collections"].items())
                {
                    Collection aCollection{ "","" };
                    aCollection.count = int(item.value()["count"]);
                    aCollection.id = item.value()["id"];
                    aCollection.create_date = item.value()["create_date"];
                    aCollection.description = item.value()["description"];
                    aCollection.modified_date = item.value()["modified_date"];
                    aCollection.name = item.value()["name"];

                    results.push_back(aCollection);
                }
            }
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        collectioncount.collection = results;
        return collectioncount;

    }
    
    CollectionCount list(Options ops) 
    {
        CollectionCount collectioncount;
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
            {"Accept-Encoding", "gzip, deflate, br"},
            {"User-Agent", "cpp API"},
            {"x-api-key", DEVELOPER_KEY},
            {"Connection", "keep-alive"}
        };
        std::string str = "/collections?skip=" + std::to_string(ops.skip) + "&take=" + std::to_string(ops.take)
            + "&order=" + ops.order + "&search=" + ops.search;
        //std::string str = "/collections?skip=0&take=20&order=ASC";
        std::vector<char> route_collection_get(str.begin(), str.end());
        route_collection_get.push_back('\0');
        const char* route_{ &route_collection_get[0] };
        std::vector<Collection> results{};
        auto res = cli.Get(route_, headers);
        if (auto res = cli.Get(route_, headers))
        {
            json Return_body = json::parse(res->body);
            APIError(Return_body, int(res->status));
            collectioncount.count = int(Return_body["count"]);
            if (!Return_body.empty()) {
                for (auto& item : Return_body["collections"].items())
                {
                    Collection aCollection{ "","" };
                    aCollection.count = int(item.value()["count"]);
                    aCollection.id = item.value()["id"];
                    aCollection.create_date = item.value()["create_date"];
                    aCollection.description = item.value()["description"];
                    aCollection.modified_date = item.value()["modified_date"];
                    aCollection.name = item.value()["name"];

                    results.push_back(aCollection);
                }
            }
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        collectioncount.collection = results;
        return collectioncount;

    }

    std::vector<Collection> get(Options ops)
    {
        using json = nlohmann::json;

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        std::string str = "/collections?skip=" + std::to_string(ops.skip) + "&take=" + std::to_string(ops.take)
            + "&order=" + ops.order + "&search=" + ops.search;
        //std::string str = "/collections?skip=0&take=20&order=ASC";
        std::vector<char> route_collection_get(str.begin(), str.end());
        route_collection_get.push_back('\0');
        const char* route_{ &route_collection_get[0] };
        std::vector<Collection> results{};
        auto res = cli.Get(route_, headers);
        if (auto res = cli.Get(route_, headers))
        {
            json Return_body = json::parse(res->body);
            APIError(Return_body, int(res->status));
            if (!Return_body.empty()) {
                for (auto& item : Return_body["collections"].items())
                {
                    Collection aCollection{ "","" };
                    aCollection.count = int(item.value()["count"]);
                    aCollection.id = item.value()["id"];
                    aCollection.create_date = item.value()["create_date"];
                    if (!item.value()["description"].is_null())
                        aCollection.description = item.value()["description"];
                    else aCollection.description = "";
                    aCollection.modified_date = item.value()["modified_date"];
                    aCollection.name = item.value()["name"];

                    results.push_back(aCollection);
                }
            }
        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";

#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }

        return results;
    }

    Collection update(Collection colToUpdate)
    {
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false);
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        httplib::Headers headers{
        {"Accept-Encoding", "gzip, deflate, br"},
        {"User-Agent", "cpp API"},
        {"x-api-key", DEVELOPER_KEY},
        {"Connection", "keep-alive"}
        };
        json JsonBody;
        JsonBody["name"] = colToUpdate.name;
        JsonBody["id"] = colToUpdate.id;
        JsonBody["description"] = colToUpdate.description;
        JsonBody["count"] = colToUpdate.count;
        JsonBody["create_date"] = colToUpdate.create_date;
        JsonBody["modified_date"] = colToUpdate.modified_date;
        std::string body = JsonBody.dump();
        std::string str = "/collection";
        std::vector<char> route_collection_get(str.begin(), str.end());
        route_collection_get.push_back('\0');
        const char* route_{ &route_collection_get[0] };
        Collection returnCollection{"",""};
        if (auto res = cli.Patch(route_, headers, body, "application/json"))
        {
            json result = json::parse(res->body);
            APIError(result, int(res->status));
            returnCollection.count = result["count"];
            returnCollection.id = result["id"];
            returnCollection.description = result["description"];
            returnCollection.name = result["name"];
            returnCollection.create_date = result["create_date"];
            returnCollection.modified_date = result["modified_date"];

        }
        else
        {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
        }
        return returnCollection;
    }

    Collection create(CollectionBase CB)
    {
        std::string name = CB.name;
        std::string note = CB.description;
        CollectionCount collectionCount;
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
        httplib::SSLClient cli(BACKEND_URL, port);
        cli.enable_server_certificate_verification(false); 
#else
        httplib::Client cli(BACKEND_URL, port);   //{"x-api-key": self.__developer_key}
#endif

        
        
        using json = nlohmann::json;
        json json_body;
        json_body["name"] = CB.name;
        json_body["description"] = CB.description;
        std::string j_body =  json_body.dump();

        char* writable = new char[j_body.size() + 1];
        std::copy(j_body.begin(), j_body.end(), writable);
        writable[j_body.size()] = '\0';

        httplib::Headers headers{
        {"x-api-key", DEVELOPER_KEY},
        };
        
        Collection ReturnCollection{ "","" };
        const char* route_customer{ "/collection" };
        
        
        if (auto res = cli.Post(route_customer, headers, j_body ,"application/json")) // working but not perfect
        {
            //cout << res->body<<"\n";
            json Return_body = json::parse(res->body);
            if (int(res->status) != 201) {
                APIError(Return_body, int(res->status));
            }
            //collectionCount.count = Return_body["count"];

            ReturnCollection.id = Return_body["id"];
            ReturnCollection.count = int(Return_body["count"]);
            ReturnCollection.description = Return_body["description"];
            ReturnCollection.name = Return_body["name"];
            ReturnCollection.create_date = Return_body["create_date"];
            ReturnCollection.modified_date = Return_body["modified_date"];
            
        }
        else {
            ErrorAPI errorAPI;
            errorAPI.error_code = "Connection error";
#ifdef CPPHTTPLIB_OPENSSL_SUPPORT
            auto result = cli.get_openssl_verify_result();
            if (result) {
                errorAPI.cert_error_string = X509_verify_cert_error_string(result);
            }
#endif
            throw errorAPI;
        }
        return ReturnCollection;
    }
    
};

class SDK
{
private:
    int port;
    std::string BACKEND_URL;
    std::string DEVELOPER_KEY;

public:
    Persons persons;
    Collections collections;
    Search search;
    Liveness liveness;
    Compare compare;
    SDK(std::string BACKEND_URL_, std::string DEVELOPER_KEY_)
    {
        BACKEND_URL = BACKEND_URL_;
        DEVELOPER_KEY = DEVELOPER_KEY_;
        port = 443; // default port
        persons.setter(BACKEND_URL, DEVELOPER_KEY,port);
        collections.setter(BACKEND_URL, DEVELOPER_KEY,port);
        search.setter(BACKEND_URL,DEVELOPER_KEY,port);
        liveness.setter(BACKEND_URL, DEVELOPER_KEY, port);
        compare.setter(BACKEND_URL, DEVELOPER_KEY, port);
    }
    SDK(std::string BACKEND_URL_, std::string DEVELOPER_KEY_,int port_)
    {
        BACKEND_URL = BACKEND_URL_;
        DEVELOPER_KEY = DEVELOPER_KEY_;
        port = port_;
        persons.setter(BACKEND_URL, DEVELOPER_KEY, port);
        collections.setter(BACKEND_URL, DEVELOPER_KEY, port);
        search.setter(BACKEND_URL, DEVELOPER_KEY, port);
        liveness.setter(BACKEND_URL, DEVELOPER_KEY, port);
        compare.setter(BACKEND_URL, DEVELOPER_KEY, port);
    }
    

    void print_SDK_info() {
        cout <<"\nSDK info\n" << BACKEND_URL << " and " << DEVELOPER_KEY << "\n";
        cout << "Port : " << port;
        cout << "\n==============\n";
    }
};

