/*
  ==============================================================================

   This file is part of the JUCE tutorials.
   Copyright (c) 2020 - Raw Material Software Limited

   The code included in this file is provided under the terms of the ISC license
   http://www.isc.org/downloads/software-support-policy/isc-license. Permission
   To use, copy, modify, and/or distribute this software for any purpose with or
   without fee is hereby granted provided that the above copyright notice and
   this permission notice appear in all copies.

   THE SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL WARRANTIES,
   WHETHER EXPRESSED OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR
   PURPOSE, ARE DISCLAIMED.

  ==============================================================================
*/

/*******************************************************************************
 The block below describes the properties of this PIP. A PIP is a short snippet
 of code that can be read by the Projucer and used to generate a JUCE project.

 BEGIN_JUCE_PIP_METADATA

 name:             PushNotificationsTutorial
 version:          2.0.0
 vendor:           JUCE
 website:          http://juce.com
 description:      Explores push notifications features.

 dependencies:     juce_audio_basics, juce_audio_devices, juce_audio_formats,
                   juce_audio_processors, juce_audio_utils, juce_core,
                   juce_data_structures, juce_events, juce_graphics,
                   juce_gui_basics, juce_gui_extra
 exporters:        xcode_mac, xcode_iphone, androidstudio

 moduleFlags:      JUCE_PUSH_NOTIFICATIONS=1

 type:             Component
 mainClass:        MainContentComponent

 useLocalCopy:     1

 END_JUCE_PIP_METADATA

*******************************************************************************/


#pragma once

//==============================================================================
class MainContentComponent   : public juce::Component,
                               private juce::ChangeListener,
                               private juce::ComponentListener,
                               private juce::PushNotifications::Listener
{
public:
    //==============================================================================
    MainContentComponent()
    {
        setupControls();
        distributeControls();

      #if JUCE_PUSH_NOTIFICATIONS
        addAndMakeVisible (headerLabel);
        addAndMakeVisible (mainTabs);
        addAndMakeVisible (sendButton);
      #else
        addAndMakeVisible (notAvailableYetLabel);
      #endif

        headerLabel.setJustificationType (juce::Justification::centred);
        notAvailableYetLabel.setJustificationType (juce::Justification::centred);

      #if JUCE_MAC
        juce::StringArray tabNames { "Params1", "Params2", "Params3", "Params4" };
      #else
        juce::StringArray tabNames { "Req. params", "Opt. params1", "Opt. params2", "Opt. params3" };
      #endif

        auto colour = getLookAndFeel().findColour (juce::ResizableWindow::backgroundColourId);
        localNotificationsTabs.addTab (tabNames[0], colour, &paramsOneView, false);
        localNotificationsTabs.addTab (tabNames[1], colour, &paramsTwoView, false);
      #if JUCE_ANDROID
        localNotificationsTabs.addTab (tabNames[2], colour, &paramsThreeView, false);
        localNotificationsTabs.addTab (tabNames[3], colour, &paramsFourView, false);
      #endif
        localNotificationsTabs.addTab ("Aux. actions", colour, &auxActionsView, false);

        mainTabs.addTab ("Local", colour, &localNotificationsTabs, false);
        mainTabs.addTab ("Remote", colour, &remoteView, false);

        auto userArea = juce::Desktop::getInstance().getDisplays().getPrimaryDisplay()->userArea;
      #if JUCE_ANDROID || JUCE_IOS
        setSize (userArea.getWidth(), userArea.getHeight());
      #else
        setSize (userArea.getWidth() / 2, userArea.getHeight() / 2);
      #endif

        sendButton.onClick = [this] { sendLocalNotification(); };
        auxActionsView.getDeliveredNotificationsButton .onClick = []
            { juce::PushNotifications::getInstance()->getDeliveredNotifications(); };
        auxActionsView.removeDeliveredNotifWithIdButton.onClick = [this]
            { juce::PushNotifications::getInstance()->removeDeliveredNotification (auxActionsView.deliveredNotifIdentifier.getText()); };
        auxActionsView.removeAllDeliveredNotifsButton  .onClick = []
            { juce::PushNotifications::getInstance()->removeAllDeliveredNotifications(); };
      #if JUCE_IOS || JUCE_MAC
        auxActionsView.getPendingNotificationsButton .onClick = []
            { juce::PushNotifications::getInstance()->getPendingLocalNotifications(); };
        auxActionsView.removePendingNotifWithIdButton.onClick = [this]
            { juce::PushNotifications::getInstance()->removePendingLocalNotification (auxActionsView.pendingNotifIdentifier.getText()); };
        auxActionsView.removeAllPendingNotifsButton  .onClick = []
            { juce::PushNotifications::getInstance()->removeAllPendingLocalNotifications(); };
      #endif

        remoteView.getDeviceTokenButton.onClick = []
        {
            juce::String token = juce::PushNotifications::getInstance()->getDeviceToken();

            DBG ("token = " + token);

            if (token.isEmpty())
                showRemoteInstructions();
            else
                juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon, "Device token", token);
        };

      #if JUCE_ANDROID
        remoteView.sendRemoteMessageButton.onClick = [this]
        {
            juce::StringPairArray data;                                         // [1]
            data.set ("key1", "value1");
            data.set ("key2", "value2");

            static int id = 100;
            juce::PushNotifications::getInstance()->sendUpstreamMessage ("XXXXXXXXXXXX", // Insert sender ID here
                                                                         "com.juce.pushnotificationstutorial",
                                                                         String (id++),
                                                                         "standardType",
                                                                         3600,
                                                                         data); // [2]
        };

        remoteView.subscribeToSportsButton    .onClick = [this]
            { juce::PushNotifications::getInstance()->subscribeToTopic ("sports"); };
        remoteView.unsubscribeFromSportsButton.onClick = [this]
            { juce::PushNotifications::getInstance()->unsubscribeFromTopic ("sports"); };
      #endif

        paramControls.accentColourButton.onClick = [this] { setupAccentColour(); };
        paramControls.ledColourButton   .onClick = [this] { setupLedColour(); };

        jassert (juce::PushNotifications::getInstance()->areNotificationsEnabled());

        juce::PushNotifications::getInstance()->addListener (this);

      #if JUCE_IOS || JUCE_MAC
        paramControls.fireInComboBox.onChange = [this] { delayNotification(); };
        juce::PushNotifications::getInstance()->requestPermissionsWithSettings (getNotificationSettings());
      #elif JUCE_ANDROID
        juce::PushNotifications::ChannelGroup cg { "demoGroup", "demo group" };
        juce::PushNotifications::getInstance()->setupChannels ({{ cg }}, getAndroidChannels());
      #endif
    }

    ~MainContentComponent() override
    {
        juce::PushNotifications::getInstance()->removeListener (this);
    }

    void paint (juce::Graphics& g) override
    {
        g.fillAll (getLookAndFeel().findColour (juce::ResizableWindow::backgroundColourId));
    }

    void resized() override
    {
        auto bounds = getLocalBounds().reduced (getWidth() / 20, getHeight() / 40);

        headerLabel.setBounds (bounds.removeFromTop (bounds.proportionOfHeight (0.1f)));
        mainTabs   .setBounds (bounds.removeFromTop (bounds.proportionOfHeight (0.8f)));
        sendButton .setBounds (bounds);

        notAvailableYetLabel.setBounds (getLocalBounds());
    }

private:
    void delayNotification()
    {
        auto repeatsAllowed = paramControls.fireInComboBox.getSelectedItemIndex() >= 6;

        paramControls.repeatButton.setEnabled (repeatsAllowed);

        if (! repeatsAllowed)
            paramControls.repeatButton.setToggleState (false, juce::NotificationType::sendNotification);
    }

    void sendLocalNotification()
    {
        juce::PushNotifications::Notification n;                            // [1]

        fillRequiredParams (n);                                             // [4]
        fillOptionalParamsOne (n);
      #if JUCE_ANDROID
        fillOptionalParamsTwo (n);
        fillOptionalParamsThree (n);
      #endif

        if (! n.isValid())                                                  // [2]
        {
          #if JUCE_IOS
            juce::String requiredFields = "identifier (from iOS 10), title, body and category";
          #elif JUCE_ANDROID
            juce::String requiredFields = "channel ID (from Android O), title, body and icon";
          #else
            juce::String requiredFields = "all required fields";
          #endif

            juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                         "Incorrect notifications setup",
                                                         "Please make sure that "
                                                         + requiredFields + " are set.");


            return;
        }

        juce::PushNotifications::getInstance()->sendLocalNotification (n);  // [3]
    }

    void fillRequiredParams      (juce::PushNotifications::Notification& n)
    {
        n.identifier = paramControls.identifierEditor.getText();            // [5.1]
        n.title = paramControls.titleEditor.getText();                      // [5.2]
        n.body = paramControls.bodyEditor.getText();                        // [5.3]
      #if JUCE_IOS
        n.category = paramControls.categoryComboBox.getText();              // [6.1]
      #elif JUCE_ANDROID || JUCE_MAC
       #if JUCE_MAC
        juce::String prefix = "images/";
        juce::String extension = ".png";
       #else
        juce::String prefix;
        juce::String extension;
       #endif
        if (paramControls.iconComboBox.getSelectedItemIndex() == 0)         // [7]
            n.icon = prefix + "ic_stat_name" + extension;
        else if (paramControls.iconComboBox.getSelectedItemIndex() == 1)
            n.icon = prefix + "ic_stat_name2" + extension;
        else if (paramControls.iconComboBox.getSelectedItemIndex() == 2)
            n.icon = prefix + "ic_stat_name3" + extension;
        else if (paramControls.iconComboBox.getSelectedItemIndex() == 3)
            n.icon = prefix + "ic_stat_name4" + extension;
        else if (paramControls.iconComboBox.getSelectedItemIndex() == 4)
            n.icon = prefix + "ic_stat_name5" + extension;
      #endif

      #if JUCE_ANDROID
        // Note: this is not strictly speaking required param, just doing it here because it is the fastest way!
        n.publicVersion = new juce::PushNotifications::Notification();      // [8]
        n.publicVersion->identifier = "blahblahblah";
        n.publicVersion->title = "Public title!";
        n.publicVersion->body  = "Public body!";
        n.publicVersion->icon  = n.icon;

        n.channelId = String (paramControls.channelIdComboBox.getSelectedItemIndex() + 1); // [6.2]
      #endif
    }

    void fillOptionalParamsOne   (juce::PushNotifications::Notification& n)
    {
        n.subtitle = paramControls.subtitleEditor.getText();                            // [1.1]
        n.badgeNumber = paramControls.badgeNumberComboBox.getSelectedItemIndex();       // [1.2]

        if (paramControls.soundToPlayComboBox.getSelectedItemIndex() > 0)
            n.soundToPlay = juce::URL (paramControls.soundToPlayComboBox.getItemText (paramControls.soundToPlayComboBox.getSelectedItemIndex())); // [1.3]

        n.properties = juce::JSON::parse (paramControls.propertiesEditor.getText());    // [1.4]

      #if JUCE_IOS || JUCE_MAC
        n.triggerIntervalSec = double (paramControls.fireInComboBox.getSelectedItemIndex() * 10);
        n.repeat = paramControls.repeatButton.getToggleState();
      #elif JUCE_ANDROID
        n.badgeIconType = (juce::PushNotifications::Notification::BadgeIconType) paramControls.badgeIconComboBox.getSelectedItemIndex(); // [2.1]
        n.tickerText  = paramControls.tickerTextEditor.getText();                       // [2.2]

        n.shouldAutoCancel = paramControls.autoCancelButton.getToggleState();           // [2.3]
        n.alertOnlyOnce = paramControls.alertOnlyOnceButton.getToggleState();           // [2.4]
      #endif

      #if JUCE_ANDROID || JUCE_MAC
        if (paramControls.actionsComboBox.getSelectedItemIndex() == 1)
        {
            juce::PushNotifications::Notification::Action a, a2;
            a .style = juce::PushNotifications::Notification::Action::button;
            a2.style = juce::PushNotifications::Notification::Action::button;
            a .title = a .identifier = "Ok";
            a2.title = a2.identifier = "Cancel";
            n.actions.add (a);
            n.actions.add (a2);
        }
        else if (paramControls.actionsComboBox.getSelectedItemIndex() == 2)
        {
            juce::PushNotifications::Notification::Action a, a2;
            a .title = a .identifier = "Input Text Here";
            a2.title = a2.identifier = "No";
            a .style = juce::PushNotifications::Notification::Action::text;
            a2.style = juce::PushNotifications::Notification::Action::button;
            a .icon = "ic_stat_name4";
            a2.icon = "ic_stat_name5";
            a.textInputPlaceholder = "placeholder text ...";
            n.actions.add (a);
            n.actions.add (a2);
        }
        else if (paramControls.actionsComboBox.getSelectedItemIndex() == 3)
        {
            juce::PushNotifications::Notification::Action a, a2;
            a .title = a .identifier = "Ok";
            a2.title = a2.identifier = "Cancel";
            a .style = juce::PushNotifications::Notification::Action::button;
            a2.style = juce::PushNotifications::Notification::Action::button;
            a .icon = "ic_stat_name4";
            a2.icon = "ic_stat_name5";
            n.actions.add (a);
            n.actions.add (a2);
        }
        else if (paramControls.actionsComboBox.getSelectedItemIndex() == 4)
        {
            juce::PushNotifications::Notification::Action a, a2;
            a .title = a .identifier = "Input Text Here";
            a2.title = a2.identifier = "No";
            a .style = juce::PushNotifications::Notification::Action::text;
            a2.style = juce::PushNotifications::Notification::Action::button;
            a .icon = "ic_stat_name4";
            a2.icon = "ic_stat_name5";
            a.textInputPlaceholder = "placeholder text ...";
            a.allowedResponses.add ("Response 1");
            a.allowedResponses.add ("Response 2");
            a.allowedResponses.add ("Response 3");
            n.actions.add (a);
            n.actions.add (a2);
        }
      #endif
    }

    void fillOptionalParamsTwo   (juce::PushNotifications::Notification& n)
    {
        using Notification = juce::PushNotifications::Notification;

        Notification::Progress progress;
        progress.max     = paramControls.progressMaxComboBox.getSelectedItemIndex() * 10;
        progress.current = paramControls.progressCurrentComboBox.getSelectedItemIndex() * 10;
        progress.indeterminate = paramControls.progressIndeterminateButton.getToggleState();

        n.progress = progress;                                  // [3.1]
        n.person   = paramControls.personEditor.getText();      // [3.2]
        n.type                 = Notification::Type                 (paramControls.categoryComboBox.getSelectedItemIndex());                    // [3.3]
        n.priority             = Notification::Priority             (paramControls.priorityComboBox.getSelectedItemIndex() - 2);                // [3.4]
        n.lockScreenAppearance = Notification::LockScreenAppearance (paramControls.lockScreenVisibilityComboBox.getSelectedItemIndex() - 1);    // [3.5]
        n.groupId = paramControls.groupIdEditor.getText();      // [3.6]
        n.groupSortKey = paramControls.sortKeyEditor.getText();
        n.groupSummary = paramControls.groupSummaryButton.getToggleState();
        n.groupAlertBehaviour = Notification::GroupAlertBehaviour (paramControls.groupAlertBehaviourComboBox.getSelectedItemIndex());
    }

    void fillOptionalParamsThree (juce::PushNotifications::Notification& n)
    {
        n.accentColour = paramControls.accentColourButton.findColour (juce::TextButton::buttonColourId, false);     // [4.1]
        n.ledColour    = paramControls.ledColourButton   .findColour (juce::TextButton::buttonColourId, false);     // [4.2]

        using Notification = juce::PushNotifications::Notification;
        Notification::LedBlinkPattern ledBlinkPattern;
        ledBlinkPattern.msToBeOn  = paramControls.ledMsToBeOnComboBox .getSelectedItemIndex() * 200;
        ledBlinkPattern.msToBeOff = paramControls.ledMsToBeOffComboBox.getSelectedItemIndex() * 200;
        n.ledBlinkPattern = ledBlinkPattern;                                                                        // [4.3]

        juce::Array<int> vibrationPattern;

        if (paramControls.vibratorMsToBeOnComboBox .getSelectedItemIndex() > 0 &&
            paramControls.vibratorMsToBeOffComboBox.getSelectedItemIndex() > 0)
        {
            vibrationPattern.add (paramControls.vibratorMsToBeOffComboBox.getSelectedItemIndex() * 500);
            vibrationPattern.add (paramControls.vibratorMsToBeOnComboBox .getSelectedItemIndex() * 500);
            vibrationPattern.add (2 * paramControls.vibratorMsToBeOffComboBox.getSelectedItemIndex() * 500);
            vibrationPattern.add (2 * paramControls.vibratorMsToBeOnComboBox .getSelectedItemIndex() * 500);
        }

        n.vibrationPattern = vibrationPattern;                                                                      // [4.4]

        n.localOnly = paramControls.localOnlyButton.getToggleState();                                               // [4.5]
        n.ongoing = paramControls.ongoingButton.getToggleState();                                                   // [4.6]
        n.timestampVisibility = Notification::TimestampVisibility (paramControls.timestampVisibilityComboBox.getSelectedItemIndex()); // [4.7]

        if (paramControls.timeoutAfterComboBox.getSelectedItemIndex() > 0)
        {
            auto index = paramControls.timeoutAfterComboBox.getSelectedItemIndex();
            n.timeoutAfterMs = index * 1000 + 4000;                                                                 // [4.8]
        }
    }

    void setupAccentColour()
    {
        auto accentColourSelector = std::make_unique<juce::ColourSelector>();

        accentColourSelector->setName ("accent colour");
        accentColourSelector->setCurrentColour (paramControls.accentColourButton.findColour (juce::TextButton::buttonColourId));
        accentColourSelector->setColour (juce::ColourSelector::backgroundColourId, juce::Colours::transparentBlack);
        accentColourSelector->setSize (200, 200);
        accentColourSelector->addComponentListener (this);
        accentColourSelector->addChangeListener (this);

        paramControls.accentColourSelector = accentColourSelector.get();

        juce::CallOutBox::launchAsynchronously (std::move (accentColourSelector), paramControls.accentColourButton.getScreenBounds(), nullptr);
    }

    void setupLedColour()
    {
        auto ledColourSelector = std::make_unique<juce::ColourSelector>();

        ledColourSelector->setName ("led colour");
        ledColourSelector->setCurrentColour (paramControls.ledColourButton.findColour (juce::TextButton::buttonColourId));
        ledColourSelector->setColour (juce::ColourSelector::backgroundColourId, juce::Colours::transparentBlack);
        ledColourSelector->setSize (200, 200);
        ledColourSelector->addComponentListener (this);
        ledColourSelector->addChangeListener (this);

        paramControls.ledColourSelector = ledColourSelector.get();

        juce::CallOutBox::launchAsynchronously (std::move (ledColourSelector), paramControls.accentColourButton.getScreenBounds(), nullptr);
    }

    void changeListenerCallback (juce::ChangeBroadcaster* source) override
    {
        if (source == paramControls.accentColourSelector)
        {
            auto c = paramControls.accentColourSelector->getCurrentColour();
            paramControls.accentColourButton.setColour (juce::TextButton::buttonColourId, c);
        }
        else if (source == paramControls.ledColourSelector)
        {
            auto c = paramControls.ledColourSelector->getCurrentColour();
            paramControls.ledColourButton.setColour (juce::TextButton::buttonColourId, c);
        }
    }

    void componentBeingDeleted (Component& component) override
    {
        if (&component == paramControls.accentColourSelector)
            paramControls.accentColourSelector = nullptr;
        else if (&component == paramControls.ledColourSelector)
            paramControls.ledColourSelector = nullptr;
    }

    void handleNotification (bool isLocalNotification, const juce::PushNotifications::Notification& n) override
    {
        juce::ignoreUnused (isLocalNotification);

        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Received notification",
                                                     "ID: " + n.identifier
                                                     + ", title: " + n.title
                                                     + ", body: " + n.body);
    }

    void handleNotificationAction (bool isLocalNotification,
                                   const juce::PushNotifications::Notification& n,
                                   const juce::String& actionIdentifier,
                                   const juce::String& optionalResponse) override
    {
        juce::ignoreUnused (isLocalNotification);

        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Received notification action",
                                                     "ID: " + n.identifier
                                                     + ", title: " + n.title
                                                     + ", body: " + n.body
                                                     + ", action: " + actionIdentifier
                                                     + ", optionalResponse: " + optionalResponse);

        juce::PushNotifications::getInstance()->removeDeliveredNotification (n.identifier);
    }

    void localNotificationDismissedByUser (const juce::PushNotifications::Notification& n) override
    {
        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Notification dismissed by a user",
                                                     "ID: " + n.identifier
                                                     + ", title: " + n.title
                                                     + ", body: " + n.body);
    }

    void deliveredNotificationsListReceived (const juce::Array<juce::PushNotifications::Notification>& notifs) override
    {
        juce::String text = "Received notifications: ";

        for (auto& n : notifs)
            text << "(" << n.identifier << ", " << n.title << ", " << n.body << "), ";

        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon, "Received notification list", text);
    }

    void pendingLocalNotificationsListReceived (const juce::Array<juce::PushNotifications::Notification>& notifs) override
    {
        juce::String text = "Pending notifications: ";

        for (auto& n : notifs)
            text << "(" << n.identifier << ", " << n.title << ", " << n.body << "), ";

        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon, "Pending notification list", text);
    }

    void deviceTokenRefreshed (const juce::String& token) override
    {
        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Device token refreshed",
                                                     token);
    }

  #if JUCE_ANDROID
    void remoteNotificationsDeleted() override
    {
        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Remote notifications deleted",
                                                     "Some of the pending messages were removed!");
    }

    void upstreamMessageSent (const juce::String& messageId) override
    {
        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Upstream message sent",
                                                     "Message id: " + messageId);
    }

    void upstreamMessageSendingError (const juce::String& messageId, const juce::String& error) override
    {
        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Upstream message sending error",
                                                     "Message id: " + messageId
                                                     + "\nerror: " + error);
    }

    static juce::Array<juce::PushNotifications::Channel> getAndroidChannels()
    {
        using Channel = juce::PushNotifications::Channel;

        Channel ch1, ch2, ch3;

        ch1.identifier = "1";
        ch1.name = "HighImportance";
        ch1.importance = juce::PushNotifications::Channel::max;
        ch1.lockScreenAppearance = juce::PushNotifications::Notification::showCompletely;
        ch1.description = "High Priority Channel for important stuff";
        ch1.groupId = "demoGroup";
        ch1.ledColour = juce::Colours::red;
        ch1.bypassDoNotDisturb = true;
        ch1.canShowBadge = true;
        ch1.enableLights = true;
        ch1.enableVibration = true;
        ch1.soundToPlay = juce::URL ("demonstrative");
        ch1.vibrationPattern = { 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200 };

        ch2.identifier = "2";
        ch2.name = "MediumImportance";
        ch2.importance = juce::PushNotifications::Channel::normal;
        ch2.lockScreenAppearance = juce::PushNotifications::Notification::showPartially;
        ch2.description = "Medium Priority Channel for standard stuff";
        ch2.groupId = "demoGroup";
        ch2.ledColour = juce::Colours::yellow;
        ch2.canShowBadge = true;
        ch2.enableLights = true;
        ch2.enableVibration = true;
        ch2.soundToPlay = juce::URL ("default_os_sound");
        ch2.vibrationPattern = { 1000, 1000 };

        ch3.identifier = "3";
        ch3.name = "LowImportance";
        ch3.importance = juce::PushNotifications::Channel::min;
        ch3.lockScreenAppearance = juce::PushNotifications::Notification::dontShow;
        ch3.description = "Low Priority Channel for silly stuff";
        ch3.groupId = "demoGroup";

        return { ch1, ch2, ch3 };
    }

  #elif JUCE_IOS || JUCE_MAC
    static juce::PushNotifications::Settings getNotificationSettings()
    {
        juce::PushNotifications::Settings settings;                             // [6]
        settings.allowAlert = true;
        settings.allowBadge = true;
        settings.allowSound = true;

      #if JUCE_IOS
        using Action   = juce::PushNotifications::Settings::Action;
        using Category = juce::PushNotifications::Settings::Category;

        Action okAction;
        okAction.identifier = "okAction";
        okAction.title = "OK!";
        okAction.style = Action::button;
        okAction.triggerInBackground = true;

        Action cancelAction;
        cancelAction.identifier = "cancelAction";
        cancelAction.title = "Cancel";
        cancelAction.style = Action::button;
        cancelAction.triggerInBackground = true;
        cancelAction.destructive = true;

        Action textAction;
        textAction.identifier = "textAction";
        textAction.title = "Enter text";
        textAction.style = Action::text;
        textAction.triggerInBackground = true;
        textAction.destructive = false;
        textAction.textInputButtonText = "Ok";
        textAction.textInputPlaceholder = "Enter text...";

        Category okCategory;
        okCategory.identifier = "okCategory";
        okCategory.actions = { okAction };

        Category okCancelCategory;
        okCancelCategory.identifier = "okCancelCategory";
        okCancelCategory.actions = { okAction, cancelAction };

        Category textCategory;
        textCategory.identifier = "textCategory";
        textCategory.actions = { textAction };
        textCategory.sendDismissAction = true;

        settings.categories = { okCategory, okCancelCategory, textCategory };   // [7]
      #endif

        return settings;
    }
  #endif

    struct RowComponent : public Component
    {
        RowComponent (juce::Label& l, Component& c, int u = 1)
            : label (l),
              editor (c),
              rowUnits (u)
        {
            addAndMakeVisible (label);
            addAndMakeVisible (editor);
        }

        void resized() override
        {
            auto bounds = getLocalBounds();
            label .setBounds (bounds.removeFromLeft (getWidth() / 3));
            editor.setBounds (bounds);
        }

        juce::Label& label;
        Component& editor;
        int rowUnits;
    };

    struct ParamControls
    {
        juce::Label        identifierLabel            { "identifierLabel", "Identifier" };
        juce::TextEditor   identifierEditor;
        juce::Label        titleLabel                 { "titleLabel", "Title" };
        juce::TextEditor   titleEditor;
        juce::Label        bodyLabel                  { "bodyLabel", "Body" };
        juce::TextEditor   bodyEditor;

        juce::Label        categoryLabel              { "categoryLabel", "Category" };
        juce::ComboBox     categoryComboBox;
        juce::Label        channelIdLabel             { "channelIdLabel", "Channel ID" };
        juce::ComboBox     channelIdComboBox;
        juce::Label        iconLabel                  { "iconLabel", "Icon" };
        juce::ComboBox     iconComboBox;

        juce::Label        subtitleLabel              { "subtitleLabel", "Subtitle" };
        juce::TextEditor   subtitleEditor;
        juce::Label        badgeNumberLabel           { "badgeNumberLabel", "BadgeNumber" };
        juce::ComboBox     badgeNumberComboBox;
        juce::Label        soundToPlayLabel           { "soundToPlayLabel", "SoundToPlay" };
        juce::ComboBox     soundToPlayComboBox;
        juce::Label        propertiesLabel            { "propertiesLabel", "Properties" };
        juce::TextEditor   propertiesEditor;
        juce::Label        fireInLabel                { "fireInLabel", "Fire in" };
        juce::ComboBox     fireInComboBox;
        juce::Label        repeatLabel                { "repeatLabel", "Repeat" };
        juce::ToggleButton repeatButton;
        juce::Label        badgeIconLabel             { "badgeIconLabel", "Badge Icon" };
        juce::ComboBox     badgeIconComboBox;
        juce::Label        tickerTextLabel            { "tickerTextLabel", "Ticker Text" };
        juce::TextEditor   tickerTextEditor;
        juce::Label        autoCancelLabel            { "autoCancelLabel", "AutoCancel" };
        juce::ToggleButton autoCancelButton;
        juce::Label        alertOnlyOnceLabel         { "alertOnlyOnceLabel", "AlertOnlyOnce" };
        juce::ToggleButton alertOnlyOnceButton;
        juce::Label        actionsLabel               { "actionsLabel", "Actions" };
        juce::ComboBox     actionsComboBox;

        juce::Label        progressMaxLabel           { "progressMaxLabel", "ProgressMax" };
        juce::ComboBox     progressMaxComboBox;
        juce::Label        progressCurrentLabel       { "progressCurrentLabel", "ProgressCurrent" };
        juce::ComboBox     progressCurrentComboBox;
        juce::Label        progressIndeterminateLabel { "progressIndeterminateLabel", "ProgressIndeterminate" };
        juce::ToggleButton progressIndeterminateButton;
        juce::Label        notifCategoryLabel         { "notifCategoryLabel", "Category" };
        juce::ComboBox     notifCategoryComboBox;
        juce::Label        priorityLabel              { "priorityLabel", "Priority" };
        juce::ComboBox     priorityComboBox;
        juce::Label        personLabel                { "personLabel", "Person" };
        juce::TextEditor   personEditor;
        juce::Label        lockScreenVisibilityLabel  { "lockScreenVisibilityLabel", "LockScreenVisibility" };
        juce::ComboBox     lockScreenVisibilityComboBox;
        juce::Label        groupIdLabel               { "groupIdLabel", "GroupID" };
        juce::TextEditor   groupIdEditor;
        juce::Label        sortKeyLabel               { "sortKeyLabel", "SortKey" };
        juce::TextEditor   sortKeyEditor;
        juce::Label        groupSummaryLabel          { "groupSummaryLabel", "GroupSummary" };
        juce::ToggleButton groupSummaryButton;
        juce::Label        groupAlertBehaviourLabel   { "groupAlertBehaviourLabel", "GroupAlertBehaviour" };
        juce::ComboBox     groupAlertBehaviourComboBox;

        juce::Label        accentColourLabel          { "accentColourLabel", "AccentColour" };
        juce::TextButton   accentColourButton;
        juce::Label        ledColourLabel             { "ledColourLabel", "LedColour" };
        juce::TextButton   ledColourButton;
        juce::Label        ledMsToBeOnLabel           { "ledMsToBeOnLabel", "LedMsToBeOn" };
        juce::ComboBox     ledMsToBeOnComboBox;
        juce::Label        ledMsToBeOffLabel          { "ledMsToBeOffLabel", "LedMsToBeOff" };
        juce::ComboBox     ledMsToBeOffComboBox;
        juce::Label        vibratorMsToBeOnLabel      { "vibratorMsToBeOnLabel", "VibrationMsToBeOn" };
        juce::ComboBox     vibratorMsToBeOnComboBox;
        juce::Label        vibratorMsToBeOffLabel     { "vibratorMsToBeOffLabel", "VibrationMsToBeOff" };
        juce::ComboBox     vibratorMsToBeOffComboBox;
        juce::Label        localOnlyLabel             { "localOnlyLabel", "LocalOnly" };
        juce::ToggleButton localOnlyButton;
        juce::Label        ongoingLabel               { "ongoingLabel", "Ongoing" };
        juce::ToggleButton ongoingButton;
        juce::Label        timestampVisibilityLabel   { "timestampVisibilityLabel", "TimestampMode" };
        juce::ComboBox     timestampVisibilityComboBox;
        juce::Label        timeoutAfterLabel          { "timeoutAfterLabel", "Timeout After Ms" };
        juce::ComboBox     timeoutAfterComboBox;

        juce::ColourSelector* accentColourSelector = nullptr;
        juce::ColourSelector* ledColourSelector    = nullptr;
    };

    void setupControls()
    {
        auto& pc = paramControls;

        juce::StringArray categories { "okCategory", "okCancelCategory", "textCategory" };

        for (auto& c : categories)
            pc.categoryComboBox.addItem (c, pc.categoryComboBox.getNumItems() + 1);
        pc.categoryComboBox.setSelectedItemIndex (0);

        for (auto i = 1; i <= 3; ++i)
            pc.channelIdComboBox.addItem (juce::String (i), i);
        pc.channelIdComboBox.setSelectedItemIndex (0);

        for (auto i = 0; i < 5; ++i)
            pc.iconComboBox.addItem ("icon" + juce::String (i + 1), i + 1);
        pc.iconComboBox.setSelectedItemIndex (0);

      #if JUCE_MAC
        pc.iconComboBox.addItem ("none", 100);
      #endif

        pc.fireInComboBox.addItem ("Now", 1);

        for (auto i = 1; i < 11; ++i)
            pc.fireInComboBox.addItem (juce::String (10 * i) + "seconds", i + 1);
        pc.fireInComboBox.setSelectedItemIndex (0);

        pc.badgeIconComboBox.addItem ("none", 1);
        pc.badgeIconComboBox.addItem ("small", 2);
        pc.badgeIconComboBox.addItem ("large", 3);
        pc.badgeIconComboBox.setSelectedItemIndex (2);

        pc.actionsComboBox.addItem ("none", 1);
        pc.actionsComboBox.addItem ("ok-cancel", 2);
        pc.actionsComboBox.addItem ("text-input", 3);
      #if JUCE_ANDROID
        pc.actionsComboBox.addItem ("ok-cancel-icons", 4);
        pc.actionsComboBox.addItem ("text-input-limited_responses", 5);
      #endif
        pc.actionsComboBox.setSelectedItemIndex (0);

        for (auto i = 0; i < 7; ++i)
            pc.badgeNumberComboBox.addItem (juce::String (i), i + 1);
        pc.badgeNumberComboBox.setSelectedItemIndex (0);

      #if JUCE_IOS
        juce::String prefix = "sounds/";
        juce::String extension = ".caf";
      #else
        juce::String prefix;
        juce::String extension;
      #endif

        pc.soundToPlayComboBox.addItem ("none", 1);
        pc.soundToPlayComboBox.addItem ("default_os_sound", 2);
        pc.soundToPlayComboBox.addItem (prefix + "demonstrative" + extension, 3);
        pc.soundToPlayComboBox.addItem (prefix + "isntit" + extension, 4);
        pc.soundToPlayComboBox.addItem (prefix + "jinglebellssms" + extension, 5);
        pc.soundToPlayComboBox.addItem (prefix + "served" + extension, 6);
        pc.soundToPlayComboBox.addItem (prefix + "solemn" + extension, 7);
        pc.soundToPlayComboBox.setSelectedItemIndex (1);

        for (auto i = 0; i < 11; ++i)
        {
            pc.progressMaxComboBox    .addItem (juce::String (i * 10) + "%", i + 1);
            pc.progressCurrentComboBox.addItem (juce::String (i * 10) + "%", i + 1);
        }

        pc.progressMaxComboBox    .setSelectedItemIndex (0);
        pc.progressCurrentComboBox.setSelectedItemIndex (0);

        pc.notifCategoryComboBox.addItem ("unspecified",     1);
        pc.notifCategoryComboBox.addItem ("alarm",           2);
        pc.notifCategoryComboBox.addItem ("call",            3);
        pc.notifCategoryComboBox.addItem ("email",           4);
        pc.notifCategoryComboBox.addItem ("error",           5);
        pc.notifCategoryComboBox.addItem ("event",           6);
        pc.notifCategoryComboBox.addItem ("message",         7);
        pc.notifCategoryComboBox.addItem ("progress",        8);
        pc.notifCategoryComboBox.addItem ("promo",           9);
        pc.notifCategoryComboBox.addItem ("recommendation", 10);
        pc.notifCategoryComboBox.addItem ("reminder",       11);
        pc.notifCategoryComboBox.addItem ("service",        12);
        pc.notifCategoryComboBox.addItem ("social",         13);
        pc.notifCategoryComboBox.addItem ("status",         14);
        pc.notifCategoryComboBox.addItem ("system",         15);
        pc.notifCategoryComboBox.addItem ("transport",      16);
        pc.notifCategoryComboBox.setSelectedItemIndex (0);

        for (auto i = -2; i < 3; ++i)
            pc.priorityComboBox.addItem (juce::String (i), i + 3);
        pc.priorityComboBox.setSelectedItemIndex (2);

        pc.lockScreenVisibilityComboBox.addItem ("don't show", 1);
        pc.lockScreenVisibilityComboBox.addItem ("show partially", 2);
        pc.lockScreenVisibilityComboBox.addItem ("show completely", 3);
        pc.lockScreenVisibilityComboBox.setSelectedItemIndex (1);

        pc.groupAlertBehaviourComboBox.addItem ("alert all", 1);
        pc.groupAlertBehaviourComboBox.addItem ("alert summary", 2);
        pc.groupAlertBehaviourComboBox.addItem ("alert children", 3);
        pc.groupAlertBehaviourComboBox.setSelectedItemIndex (0);

        pc.timeoutAfterComboBox.addItem ("No timeout", 1);

        for (auto i = 0; i < 10; ++i)
        {
            pc.ledMsToBeOnComboBox      .addItem (juce::String (i * 200) + "ms", i + 1);
            pc.ledMsToBeOffComboBox     .addItem (juce::String (i * 200) + "ms", i + 1);
            pc.vibratorMsToBeOnComboBox .addItem (juce::String (i * 500) + "ms", i + 1);
            pc.vibratorMsToBeOffComboBox.addItem (juce::String (i * 500) + "ms", i + 1);
            pc.timeoutAfterComboBox     .addItem (juce::String (5000 + 1000 * i) + "ms", i + 2);
        }

        pc.ledMsToBeOnComboBox      .setSelectedItemIndex (5);
        pc.ledMsToBeOffComboBox     .setSelectedItemIndex (5);
        pc.vibratorMsToBeOnComboBox .setSelectedItemIndex (0);
        pc.vibratorMsToBeOffComboBox.setSelectedItemIndex (0);
        pc.timeoutAfterComboBox     .setSelectedItemIndex (0);

        pc.timestampVisibilityComboBox.addItem ("off", 1);
        pc.timestampVisibilityComboBox.addItem ("on", 2);
        pc.timestampVisibilityComboBox.addItem ("chronometer", 3);
        pc.timestampVisibilityComboBox.addItem ("count down", 4);
        pc.timestampVisibilityComboBox.setSelectedItemIndex (1);
    }

    void distributeControls()
    {
        auto& pc = paramControls;

        paramsOneView.addRowComponent (new RowComponent (pc.identifierLabel, pc.identifierEditor));
        paramsOneView.addRowComponent (new RowComponent (pc.titleLabel, pc.titleEditor));
        paramsOneView.addRowComponent (new RowComponent (pc.bodyLabel, pc.bodyEditor, 4));
      #if JUCE_IOS
        paramsOneView.addRowComponent (new RowComponent (pc.categoryLabel, pc.categoryComboBox));
      #elif JUCE_ANDROID
        paramsOneView.addRowComponent (new RowComponent (pc.channelIdLabel, pc.channelIdComboBox));
      #endif
      #if JUCE_ANDROID || JUCE_MAC
        paramsOneView.addRowComponent (new RowComponent (pc.iconLabel, pc.iconComboBox));
      #endif

        paramsTwoView.addRowComponent (new RowComponent (pc.subtitleLabel, pc.subtitleEditor));
      #if ! JUCE_MAC
        paramsTwoView.addRowComponent (new RowComponent (pc.badgeNumberLabel, pc.badgeNumberComboBox));
      #endif
        paramsTwoView.addRowComponent (new RowComponent (pc.soundToPlayLabel, pc.soundToPlayComboBox));
        paramsTwoView.addRowComponent (new RowComponent (pc.propertiesLabel, pc.propertiesEditor, 3));
      #if JUCE_IOS || JUCE_MAC
        paramsTwoView.addRowComponent (new RowComponent (pc.fireInLabel, pc.fireInComboBox));
        paramsTwoView.addRowComponent (new RowComponent (pc.repeatLabel, pc.repeatButton));
      #elif JUCE_ANDROID
        paramsTwoView.addRowComponent (new RowComponent (pc.badgeIconLabel, pc.badgeIconComboBox));
        paramsTwoView.addRowComponent (new RowComponent (pc.tickerTextLabel, pc.tickerTextEditor));
        paramsTwoView.addRowComponent (new RowComponent (pc.autoCancelLabel, pc.autoCancelButton));
        paramsTwoView.addRowComponent (new RowComponent (pc.alertOnlyOnceLabel, pc.alertOnlyOnceButton));
      #endif
      #if JUCE_ANDROID || JUCE_MAC
        paramsTwoView.addRowComponent (new RowComponent (pc.actionsLabel, pc.actionsComboBox));
      #endif
      #if JUCE_ANDROID
        paramsThreeView.addRowComponent (new RowComponent (pc.progressMaxLabel, pc.progressMaxComboBox));
        paramsThreeView.addRowComponent (new RowComponent (pc.progressCurrentLabel, pc.progressCurrentComboBox));
        paramsThreeView.addRowComponent (new RowComponent (pc.progressIndeterminateLabel, pc.progressIndeterminateButton));
        paramsThreeView.addRowComponent (new RowComponent (pc.categoryLabel, pc.categoryComboBox));
        paramsThreeView.addRowComponent (new RowComponent (pc.priorityLabel, pc.priorityComboBox));
        paramsThreeView.addRowComponent (new RowComponent (pc.personLabel, pc.personEditor));
        paramsThreeView.addRowComponent (new RowComponent (pc.lockScreenVisibilityLabel, pc.lockScreenVisibilityComboBox));
        paramsThreeView.addRowComponent (new RowComponent (pc.groupIdLabel, pc.groupIdEditor));
        paramsThreeView.addRowComponent (new RowComponent (pc.sortKeyLabel, pc.sortKeyEditor));
        paramsThreeView.addRowComponent (new RowComponent (pc.groupSummaryLabel, pc.groupSummaryButton));
        paramsThreeView.addRowComponent (new RowComponent (pc.groupAlertBehaviourLabel, pc.groupAlertBehaviourComboBox));
        paramsFourView .addRowComponent (new RowComponent (pc.accentColourLabel, pc.accentColourButton));
        paramsFourView .addRowComponent (new RowComponent (pc.ledColourLabel, pc.ledColourButton));
        paramsFourView .addRowComponent (new RowComponent (pc.ledMsToBeOffLabel, pc.ledMsToBeOffComboBox));
        paramsFourView .addRowComponent (new RowComponent (pc.ledMsToBeOnLabel, pc.ledMsToBeOnComboBox));
        paramsFourView .addRowComponent (new RowComponent (pc.vibratorMsToBeOffLabel, pc.vibratorMsToBeOffComboBox));
        paramsFourView .addRowComponent (new RowComponent (pc.vibratorMsToBeOnLabel, pc.vibratorMsToBeOnComboBox));
        paramsFourView .addRowComponent (new RowComponent (pc.localOnlyLabel, pc.localOnlyButton));
        paramsFourView .addRowComponent (new RowComponent (pc.ongoingLabel, pc.ongoingButton));
        paramsFourView .addRowComponent (new RowComponent (pc.timestampVisibilityLabel, pc.timestampVisibilityComboBox));
        paramsFourView .addRowComponent (new RowComponent (pc.timeoutAfterLabel, pc.timeoutAfterComboBox));
      #endif
    }

    struct ParamsView   : public Component
    {
        ParamsView()
        {
            // For now, to be able to dismiss mobile keyboard.
            setWantsKeyboardFocus (true);
        }

        void addRowComponent (RowComponent *rc)
        {
            rowComponents.add (rc);
            addAndMakeVisible (rc);
        }

        void resized() override
        {
            auto totalRowUnits = 0;

            for (auto &rc : rowComponents)
                totalRowUnits += rc->rowUnits;

            auto rowHeight = getHeight() / totalRowUnits;

            auto bounds = getLocalBounds();

            for (auto &rc : rowComponents)
                rc->setBounds (bounds.removeFromTop (rc->rowUnits * rowHeight));

            auto* last = rowComponents[rowComponents.size() - 1];
            last->setBounds (last->getBounds().withHeight (getHeight() - last->getY()));
        }

    private:
        juce::OwnedArray<RowComponent> rowComponents;
    };

    struct AuxActionsView   : public Component
    {
        AuxActionsView()
        {
            addAndMakeVisible (getDeliveredNotificationsButton);
            addAndMakeVisible (removeDeliveredNotifWithIdButton);
            addAndMakeVisible (deliveredNotifIdentifier);
            addAndMakeVisible (removeAllDeliveredNotifsButton);
          #if JUCE_IOS || JUCE_MAC
            addAndMakeVisible (getPendingNotificationsButton);
            addAndMakeVisible (removePendingNotifWithIdButton);
            addAndMakeVisible (pendingNotifIdentifier);
            addAndMakeVisible (removeAllPendingNotifsButton);
          #endif

            // For now, to be able to dismiss mobile keyboard.
            setWantsKeyboardFocus (true);
        }

        void resized() override
        {
            auto columnWidth = getWidth();

            auto rowHeight = getHeight() / 6;

            auto bounds = getLocalBounds();

            getDeliveredNotificationsButton .setBounds (bounds.removeFromTop (rowHeight));

            auto rowBounds = bounds.removeFromTop (rowHeight);
            removeDeliveredNotifWithIdButton.setBounds (rowBounds.removeFromLeft (columnWidth / 2));
            deliveredNotifIdentifier        .setBounds (rowBounds);

            removeAllDeliveredNotifsButton  .setBounds (bounds.removeFromTop (rowHeight));

          #if JUCE_IOS || JUCE_MAC
            getPendingNotificationsButton .setBounds (bounds.removeFromTop (rowHeight));

            rowBounds = bounds.removeFromTop (rowHeight);
            removePendingNotifWithIdButton.setBounds (rowBounds.removeFromLeft (columnWidth / 2));
            pendingNotifIdentifier        .setBounds (rowBounds);

            removeAllPendingNotifsButton  .setBounds (bounds.removeFromTop (rowHeight));
          #endif
        }

        juce::TextButton getDeliveredNotificationsButton  { "Get Delivered Notifications" };
        juce::TextButton removeDeliveredNotifWithIdButton { "Remove Delivered Notif With ID:" };
        juce::TextEditor deliveredNotifIdentifier;
        juce::TextButton removeAllDeliveredNotifsButton   { "Remove All Delivered Notifs" };
        juce::TextButton getPendingNotificationsButton    { "Get Pending Notifications" };
        juce::TextButton removePendingNotifWithIdButton   { "Remove Pending Notif With ID:" };
        juce::TextEditor pendingNotifIdentifier;
        juce::TextButton removeAllPendingNotifsButton     { "Remove All Pending Notifs" };
    };

    struct RemoteView   : public Component
    {
        RemoteView()
        {
            addAndMakeVisible (getDeviceTokenButton);
          #if JUCE_ANDROID
            addAndMakeVisible (sendRemoteMessageButton);
            addAndMakeVisible (subscribeToSportsButton);
            addAndMakeVisible (unsubscribeFromSportsButton);
          #endif
        }

        void resized()
        {
            auto rowSize = getHeight () / 10;

            auto bounds = getLocalBounds().reduced (getWidth() / 10, getHeight() / 10);

            bounds.removeFromTop (2 * rowSize);

            getDeviceTokenButton       .setBounds (bounds.removeFromTop (rowSize));
            sendRemoteMessageButton    .setBounds (bounds.removeFromTop (rowSize));
            subscribeToSportsButton    .setBounds (bounds.removeFromTop (rowSize));
            unsubscribeFromSportsButton.setBounds (bounds.removeFromTop (rowSize));
        }

        juce::TextButton getDeviceTokenButton        { "GetDeviceToken" };
        juce::TextButton sendRemoteMessageButton     { "SendRemoteMessage" };
        juce::TextButton subscribeToSportsButton     { "SubscribeToSports" };
        juce::TextButton unsubscribeFromSportsButton { "UnsubscribeFromSports" };
    };

    struct DemoTabbedComponent  : public juce::TabbedComponent
    {
        explicit DemoTabbedComponent (juce::TabbedButtonBar::Orientation orientation)
            : TabbedComponent (orientation)
        {
        }

        void currentTabChanged (int, const juce::String& newCurrentTabName) override
        {
            if (! showedRemoteInstructions && newCurrentTabName == "Remote")
            {
                MainContentComponent::showRemoteInstructions();

                showedRemoteInstructions = true;
            }

        }

    private:
        bool showedRemoteInstructions = false;
    };

    static void showRemoteInstructions()
    {
      #if JUCE_IOS || JUCE_MAC
        juce::NativeMessageBox::showMessageBoxAsync (juce::AlertWindow::InfoIcon,
                                                     "Remote Notifications instructions",
                                                     "In order to be able to test remote notifications "
                                                     "ensure that the app is signed and that you register "
                                                     "the bundle ID for remote notifications in "
                                                     "Apple Developer Center.");
      #endif
    }

    juce::Label headerLabel { "headerLabel", "Push Notifications Tutorial" };
    ParamControls paramControls;
    ParamsView paramsOneView;
    ParamsView paramsTwoView;
    ParamsView paramsThreeView;
    ParamsView paramsFourView;
    AuxActionsView auxActionsView;
    juce::TabbedComponent localNotificationsTabs { juce::TabbedButtonBar::TabsAtTop };
    RemoteView remoteView;
    DemoTabbedComponent mainTabs     { juce::TabbedButtonBar::TabsAtTop };
    juce::TextButton sendButton      { "Send!" };
    juce::Label notAvailableYetLabel { "notAvailableYetLabel",
                                       "Push Notifications feature is not available on this platform yet!" };

    //==============================================================================
    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (MainContentComponent)
};
