/*
  ==============================================================================

   This file is part of the JUCE tutorials.
   Copyright (c) 2020 - Raw Material Software Limited

   The code included in this file is provided under the terms of the ISC license
   http://www.isc.org/downloads/software-support-policy/isc-license. Permission
   To use, copy, modify, and/or distribute this software for any purpose with or
   without fee is hereby granted provided that the above copyright notice and
   this permission notice appear in all copies.

   THE SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL WARRANTIES,
   WHETHER EXPRESSED OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR
   PURPOSE, ARE DISCLAIMED.

  ==============================================================================
*/

/*******************************************************************************
 The block below describes the properties of this PIP. A PIP is a short snippet
 of code that can be read by the Projucer and used to generate a JUCE project.

 BEGIN_JUCE_PIP_METADATA

 name:             RandomTutorial
 version:          2.0.0
 vendor:           JUCE
 website:          http://juce.com
 description:      Explores random use cases.

 dependencies:     juce_core, juce_data_structures, juce_events, juce_graphics,
                   juce_gui_basics
 exporters:        xcode_mac, vs2019, linux_make, xcode_iphone, androidstudio

 type:             Component
 mainClass:        MainContentComponent

 useLocalCopy:     1

 END_JUCE_PIP_METADATA

*******************************************************************************/


#pragma once

//==============================================================================
//! [all]
class MainContentComponent   : public juce::Component
{
public:
    MainContentComponent()
    {
        colours.add (juce::Colours::white);
        weights.add (0.7f);

        colours.add (juce::Colours::orange);
        weights.add (0.25f);

        colours.add (juce::Colours::lightgreen);
        weights.add (0.05f);

        setSize (400, 400);
    }

    void paint (juce::Graphics& g) override
    {
        g.fillAll (juce::Colours::black);

        juce::Random random;
        juce::Rectangle<float> rect (0.0f, 0.0f, 5.0f, 5.0f);

        auto sumOfWeights = sumFloatArray (weights); // [1]

        for (auto i = 0; i < 1000; ++i)
        {
            g.setColour (colourAtQuantile (random.nextFloat() * sumOfWeights)); // [2]

            rect.setCentre ((float) random.nextInt (getWidth()),
                            (float) random.nextInt (getHeight()));

            g.drawEllipse (rect, 1.0f);
        }
    }

private:
    static float sumFloatArray (const juce::Array<float>& values)
    {
        auto sum = 0.0f;

        for (auto value : values)
            sum += value;

        return sum;
    }

    static int indexOfQuantile (const juce::Array<float>& weights, float quantile)
    {
        auto runningTotal = 0.0f; // [4]

        for (auto weight : weights)
        {
            runningTotal += weight;

            if (quantile < runningTotal) // [5]
                return weights.indexOf (weight);
        }

        return -1;
    }

    juce::Colour colourAtQuantile (float quantile) const  // [3]
    {
        auto index = indexOfQuantile (weights, quantile);
        return index < 0 ? juce::Colours::transparentBlack : colours[index];
    }

    //==============================================================================
    juce::Array<juce::Colour> colours;
    juce::Array<float> weights;

    //==============================================================================
    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (MainContentComponent)
};
//! [all]
