/*
  ==============================================================================

   This file is part of the JUCE tutorials.
   Copyright (c) 2020 - Raw Material Software Limited

   The code included in this file is provided under the terms of the ISC license
   http://www.isc.org/downloads/software-support-policy/isc-license. Permission
   To use, copy, modify, and/or distribute this software for any purpose with or
   without fee is hereby granted provided that the above copyright notice and
   this permission notice appear in all copies.

   THE SOFTWARE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL WARRANTIES,
   WHETHER EXPRESSED OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR
   PURPOSE, ARE DISCLAIMED.

  ==============================================================================
*/

/*******************************************************************************
 The block below describes the properties of this PIP. A PIP is a short snippet
 of code that can be read by the Projucer and used to generate a JUCE project.

 BEGIN_JUCE_PIP_METADATA

 name:             MultiOutSynthTutorial
 version:          1.0.0
 vendor:           JUCE
 website:          http://juce.com
 description:      Multi-out synth audio plugin.

 dependencies:     juce_audio_basics, juce_audio_devices, juce_audio_formats,
                   juce_audio_plugin_client, juce_audio_processors,
                   juce_audio_utils, juce_core, juce_data_structures,
                   juce_events, juce_graphics, juce_gui_basics, juce_gui_extra
 exporters:        xcode_mac, vs2019, linux_make

 type:             AudioProcessor
 mainClass:        MultiOutSynth

 useLocalCopy:     1

 END_JUCE_PIP_METADATA

*******************************************************************************/


#pragma once

extern const char* singing_ogg;
const int          singing_oggSize = 15354;

//==============================================================================
class MultiOutSynth  : public juce::AudioProcessor
{
public:
//! [enum]
    enum
    {
        maxMidiChannel = 16,
        maxNumberOfVoices = 5
    };
//! [enum]

    //==============================================================================
//! [constructor]
    MultiOutSynth()
        : AudioProcessor (BusesProperties()
                          .withOutput ("Output #1",  juce::AudioChannelSet::stereo(), true)
                          .withOutput ("Output #2",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #3",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #4",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #5",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #6",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #7",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #8",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #9",  juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #10", juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #11", juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #12", juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #13", juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #14", juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #15", juce::AudioChannelSet::stereo(), false)
                          .withOutput ("Output #16", juce::AudioChannelSet::stereo(), false))   // [1]
    {
        // initialize other stuff (not related to buses)
        formatManager.registerBasicFormats();                                                   // [2]

        for (auto midiChannel = 0; midiChannel < maxMidiChannel; ++midiChannel)                 // [3]
        {
            synth.add (new juce::Synthesiser());

            for (auto i = 0; i < maxNumberOfVoices; ++i)
                synth[midiChannel]->addVoice (new juce::SamplerVoice());                        // [4]
        }

        loadNewSample (juce::MemoryBlock (singing_ogg, singing_oggSize));                       // [5]
    }
//! [constructor]

    //==============================================================================
//! [canAddBus]
    bool canAddBus    (bool isInput) const override   { return (! isInput && getBusCount (false) < maxMidiChannel); }
    bool canRemoveBus (bool isInput) const override   { return (! isInput && getBusCount (false) > 1); }
//! [canAddBus]

    //==============================================================================
//! [prepareToPlay]
    void prepareToPlay (double newSampleRate, int samplesPerBlock) override
    {
        juce::ignoreUnused (samplesPerBlock);

        for (auto midiChannel = 0; midiChannel < maxMidiChannel; ++midiChannel)
            synth[midiChannel]->setCurrentPlaybackSampleRate (newSampleRate);
    }
//! [prepareToPlay]

    void releaseResources() override {}

//! [processBlock]
    void processBlock (juce::AudioBuffer<float>& buffer, juce::MidiBuffer& midiBuffer) override
    {
        auto busCount = getBusCount (false);                // [11]

        for (auto busNr = 0; busNr < busCount; ++busNr)     // [12]
        {
            auto midiChannelBuffer = filterMidiMessagesForChannel (midiBuffer, busNr + 1);
            auto audioBusBuffer = getBusBuffer (buffer, false, busNr);

            synth [busNr]->renderNextBlock (audioBusBuffer, midiChannelBuffer, 0, audioBusBuffer.getNumSamples()); // [13]
        }
    }
//! [processBlock]

    //==============================================================================
    juce::AudioProcessorEditor* createEditor() override          { return new juce::GenericAudioProcessorEditor (*this); }
    bool hasEditor() const override                              { return true; }

    //==============================================================================
    const juce::String getName() const override                  { return "Multi Out Synth"; }
    bool acceptsMidi() const override                            { return false; }
    bool producesMidi() const override                           { return false; }
    double getTailLengthSeconds() const override                 { return 0; }
    int getNumPrograms() override                                { return 1; }
    int getCurrentProgram() override                             { return 0; }
    void setCurrentProgram (int) override                        {}
    const juce::String getProgramName (int) override             { return {}; }
    void changeProgramName (int, const juce::String&) override   {}

    //==============================================================================
    void getStateInformation (juce::MemoryBlock&) override {}
    void setStateInformation (const void*, int) override {}

private:
    //==============================================================================
//! [filterMidiMessagesForChannel]
    static juce::MidiBuffer filterMidiMessagesForChannel (const juce::MidiBuffer& input, int channel)
    {
        juce::MidiBuffer output;

        for (auto metadata : input)     // [14]
        {
            auto message = metadata.getMessage();

            if (message.getChannel() == channel)
                output.addEvent (message, metadata.samplePosition);
        }

        return output;                  // [15]
    }
//! [filterMidiMessagesForChannel]

//! [loadNewSample]
    void loadNewSample (const juce::MemoryBlock& sampleData)
    {
        auto soundBuffer = std::make_unique<juce::MemoryInputStream> (sampleData, false);   // [6]
        std::unique_ptr<juce::AudioFormatReader> formatReader (formatManager.findFormatForFileExtension ("ogg")->createReaderFor (soundBuffer.release(), true));

        juce::BigInteger midiNotes;
        midiNotes.setRange (0, 126, true);
        juce::SynthesiserSound::Ptr newSound = new juce::SamplerSound ("Voice", *formatReader, midiNotes, 0x40, 0.0, 0.0, 10.0); // [7]

        for (auto channel = 0; channel < maxMidiChannel; ++channel)             // [8]
            synth[channel]->removeSound (0);

        sound = newSound;                                                       // [9]

        for (auto channel = 0; channel < maxMidiChannel; ++channel)             // [10]
            synth[channel]->addSound (sound);
    }
//! [loadNewSample]

//! [members]
    //==============================================================================
    juce::AudioFormatManager formatManager;
    juce::OwnedArray<juce::Synthesiser> synth;
    juce::SynthesiserSound::Ptr sound;

    //==============================================================================
    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (MultiOutSynth)
};
//! [members]

static const unsigned char audioFileData[] =
{ 79,103,103,83,0,2,0,0,0,0,0,0,0,0,139,173,239,94,0,0,0,0,176,225,157,76,1,30,1,118,111,114,98,105,115,0,0,0,0,2,68,172,0,0,0,0,0,0,128,56,1,0,0,0,0,0,184,1,79,103,103,83,0,0,0,0,0,0,0,0,0,0,139,173,239,94,1,0,0,0,125,71,104,21,15,59,255,255,255,255,255,
255,255,255,255,255,255,255,255,169,3,118,111,114,98,105,115,43,0,0,0,88,105,112,104,46,79,114,103,32,108,105,98,86,111,114,98,105,115,32,73,32,50,48,49,50,48,50,48,51,32,40,79,109,110,105,112,114,101,115,101,110,116,41,0,0,0,0,1,5,118,111,114,98,105,
115,33,66,67,86,1,0,0,1,0,24,99,84,41,70,153,82,210,74,137,25,115,148,49,70,153,98,146,74,137,165,132,22,66,72,157,115,20,83,169,57,215,156,107,172,185,181,32,132,16,26,83,80,41,5,153,82,142,82,105,25,99,144,41,5,153,82,16,75,73,37,116,18,58,39,157,99,
16,91,73,193,214,152,107,139,65,182,28,132,13,154,82,76,41,196,148,82,138,66,8,25,83,140,41,197,148,82,74,66,7,37,116,14,58,230,28,83,142,74,40,65,184,156,115,171,181,150,150,99,139,169,116,146,74,231,36,100,76,66,72,41,133,146,74,7,165,83,78,66,72,53,
150,214,82,41,29,115,82,82,106,65,232,32,132,16,66,182,32,132,13,130,208,144,85,0,0,1,0,192,64,16,26,178,10,0,80,0,0,16,138,161,24,138,2,132,134,172,2,0,50,0,0,4,160,40,142,226,40,142,35,57,146,99,73,22,16,26,178,10,0,0,2,0,16,0,0,192,112,20,73,145,20,
201,177,36,75,210,44,75,211,68,81,85,125,213,54,85,85,246,117,93,215,117,93,215,117,32,52,100,21,0,0,1,0,64,72,167,153,165,26,32,194,12,100,24,8,13,89,5,0,32,0,0,0,70,40,194,16,3,66,67,86,1,0,0,1,0,0,98,40,57,136,38,180,230,124,115,142,131,102,57,104,
42,197,230,116,112,34,213,230,73,110,42,230,230,156,115,206,57,39,155,115,198,56,231,156,115,138,114,102,49,104,38,180,230,156,115,18,131,102,41,104,38,180,230,156,115,158,196,230,65,107,170,180,230,156,115,198,57,167,131,113,70,24,231,156,115,154,180,
230,65,106,54,214,230,156,115,22,180,166,57,106,46,197,230,156,115,34,229,230,73,109,46,213,230,156,115,206,57,231,156,115,206,57,231,156,115,170,23,167,115,112,78,56,231,156,115,162,246,230,90,110,66,23,231,156,115,62,25,167,123,115,66,56,231,156,115,
206,57,231,156,115,206,57,231,156,115,130,208,144,85,0,0,16,0,0,65,24,54,134,113,167,32,72,159,163,129,24,69,136,105,200,164,7,221,163,195,36,104,12,114,10,169,71,163,163,145,82,234,32,148,84,198,73,41,157,32,52,100,21,0,0,8,0,0,33,132,20,82,72,33,133,
20,82,72,33,133,20,82,136,33,134,24,98,200,41,167,156,130,10,42,169,164,162,138,50,202,44,179,204,50,203,44,179,204,50,235,176,179,206,58,236,48,196,16,67,12,173,180,18,75,77,181,213,88,99,173,185,231,156,107,14,210,90,105,173,181,214,74,41,165,148,82,
74,41,8,13,89,5,0,128,0,0,16,8,25,100,144,65,70,33,133,20,82,136,33,166,156,114,202,41,168,160,2,66,67,86,1,0,128,0,0,2,0,0,0,60,201,115,68,71,116,68,71,116,68,71,116,68,71,116,68,199,115,60,71,148,68,73,148,68,73,180,76,203,212,76,79,21,85,213,149,93,
91,214,101,221,246,109,97,23,118,221,247,117,223,247,117,227,215,133,97,89,150,101,89,150,101,89,150,101,89,150,101,89,150,101,89,130,208,144,85,0,0,8,0,0,128,16,66,8,33,133,20,82,72,33,165,24,99,204,49,231,160,147,80,66,32,52,100,21,0,0,8,0,32,0,0,0,
192,81,28,197,113,36,71,114,36,201,146,44,73,147,52,75,179,60,205,211,60,77,244,68,81,20,77,211,84,69,87,116,69,221,180,69,217,148,77,215,116,77,217,116,85,89,181,93,89,182,109,217,214,109,95,150,109,223,247,125,223,247,125,223,247,125,223,247,125,223,
247,117,29,8,13,89,5,0,72,0,0,232,72,142,164,72,138,164,72,142,227,56,146,36,1,161,33,171,0,0,25,0,0,1,0,40,138,163,56,142,227,72,146,36,73,150,164,73,158,229,89,162,102,106,166,103,122,170,168,2,161,33,171,0,0,64,0,0,1,0,0,0,0,0,40,154,226,41,166,226,
41,162,226,57,162,35,74,162,101,90,162,166,106,174,40,155,178,235,186,174,235,186,174,235,186,174,235,186,174,235,186,174,235,186,174,235,186,174,235,186,174,235,186,174,235,186,174,235,186,174,235,186,46,16,26,178,10,0,144,0,0,208,145,28,201,145,28,
73,145,20,73,145,28,201,1,66,67,86,1,0,50,0,0,2,0,112,12,199,144,20,201,177,44,75,211,60,205,211,60,77,244,68,79,244,76,79,21,93,209,5,66,67,86,1,0,128,0,0,2,0,0,0,0,0,48,36,195,82,44,71,115,52,73,148,84,75,181,84,77,181,84,75,21,85,79,85,85,85,85,85,
85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,85,53,77,211,52,77,32,52,100,37,0,16,5,0,64,57,108,177,230,222,27,97,152,114,20,115,105,140,83,142,106,80,145,66,202,89,13,42,66,10,49,137,189,85,204,49,39,49,199,206,49,
230,164,229,156,49,132,24,180,154,59,167,20,115,146,2,161,33,43,4,128,208,12,0,135,227,0,146,102,1,146,165,1,0,0,0,0,0,0,128,164,105,128,230,121,128,230,121,0,0,0,0,0,0,0,32,105,26,160,121,30,160,121,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,142,166,1,154,231,1,154,231,1,0,0,0,0,0,0,128,230,121,128,39,154,128,39,138,0,0,0,0,0,0,0,96,121,30,224,137,30,224,137,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,142,166,1,154,231,1,154,39,2,0,0,0,0,0,0,128,229,121,128,103,138,128,231,137,0,0,0,0,0,0,0,160,121,30,224,137,34,224,137,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,0,0,128,0,7,0,128,0,11,161,208,144,21,1,64,156,0,
128,67,113,44,9,0,0,28,199,177,44,0,0,112,28,201,178,0,0,192,178,44,207,3,0,0,203,178,60,15,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,192,128,3,0,64,128,9,101,
160,208,144,149,0,64,20,0,128,65,49,52,13,200,178,101,1,151,101,1,52,13,160,105,0,79,4,120,30,64,53,1,128,0,0,128,2,7,0,128,0,27,52,37,22,7,40,52,100,37,0,16,5,0,96,80,20,75,178,44,207,131,166,105,154,40,66,211,52,77,20,161,105,158,103,154,208,52,207,
51,77,136,162,231,153,38,60,207,243,76,19,166,41,138,170,10,68,81,85,5,0,0,20,56,0,0,4,216,160,41,177,56,64,161,33,43,1,128,144,0,0,131,163,88,150,166,121,158,231,137,162,105,170,42,52,205,243,68,81,20,77,211,52,85,21,154,230,121,162,40,138,166,105,154,
170,10,77,243,60,81,20,69,211,84,85,85,133,166,121,158,40,138,162,105,170,170,171,194,243,68,81,52,77,211,52,85,213,117,225,121,162,104,138,166,105,154,170,234,186,16,69,81,52,77,211,84,85,215,117,93,32,138,166,105,154,170,234,186,174,11,68,209,52,77,
85,85,93,87,150,129,40,154,166,105,170,170,235,202,50,48,77,211,84,85,215,117,93,89,6,152,166,170,186,174,235,202,50,64,85,93,215,117,101,89,150,1,170,170,170,174,43,203,178,12,112,93,215,117,93,89,182,109,0,174,235,186,178,108,219,2,0,0,14,28,0,0,2,
140,160,147,140,42,139,176,209,132,11,15,64,161,33,43,2,128,40,0,0,192,24,166,20,83,202,48,38,33,164,16,26,198,36,132,20,66,37,37,165,148,74,169,32,164,82,82,41,21,132,84,82,42,37,163,146,82,106,41,101,16,82,41,41,149,10,66,41,165,149,84,0,0,216,129,
3,0,216,129,133,80,104,200,74,0,32,15,0,128,32,68,41,198,24,115,78,74,169,20,99,206,57,39,165,84,138,49,231,156,147,82,50,198,152,115,206,73,41,25,99,204,57,231,164,148,140,57,231,156,115,82,74,198,156,115,206,57,41,165,115,206,57,231,156,148,82,74,231,
156,115,78,74,41,37,132,206,57,39,165,148,210,57,231,156,19,0,0,84,224,0,0,16,96,163,200,230,4,35,65,133,134,172,4,0,82,1,0,12,142,99,89,154,166,105,158,103,138,154,100,105,154,231,121,158,40,154,166,38,73,154,230,121,158,39,138,166,201,243,60,79,20,
69,209,52,85,147,231,121,158,40,138,162,105,170,42,215,21,69,209,52,77,85,85,85,178,44,138,162,104,154,170,170,170,48,77,211,84,85,87,117,93,152,166,41,170,170,171,202,46,100,217,52,85,213,117,101,25,182,109,154,170,234,186,178,12,84,87,85,93,215,150,
129,171,170,170,108,218,178,0,0,240,4,7,0,160,2,27,86,71,56,41,26,11,44,52,100,37,0,144,1,0,64,16,130,144,82,10,33,165,20,66,74,41,132,148,82,8,9,0,0,24,112,0,0,8,48,161,12,20,26,178,18,0,72,5,0,0,12,145,82,74,41,165,148,210,56,37,165,148,82,74,41,165,
113,76,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,
165,148,82,74,41,165,148,82,74,41,165,148,82,74,41,5,0,46,85,56,0,232,62,216,176,58,194,73,209,88,96,161,33,43,1,128,84,0,0,192,24,133,24,131,80,74,107,21,66,140,57,39,165,165,214,42,132,24,115,78,74,74,173,229,140,57,7,33,165,214,98,203,157,115,12,66,
41,173,197,216,83,233,156,148,148,90,139,177,167,20,58,42,41,181,22,91,239,189,164,146,90,107,45,198,222,123,10,41,212,214,90,140,189,247,86,83,107,45,198,26,123,239,57,182,18,75,172,49,246,222,123,143,181,197,216,98,236,189,247,30,91,75,181,229,88,0,
0,102,131,3,0,68,130,13,171,35,156,20,141,5,22,26,178,18,0,8,9,0,32,140,81,74,41,198,156,115,206,57,231,164,148,140,49,230,28,132,16,66,8,161,148,146,49,199,156,131,16,66,8,33,148,82,50,230,156,131,16,66,8,37,132,82,74,198,156,131,14,66,8,37,132,82,82,
234,156,115,16,66,8,161,132,80,74,41,157,115,14,66,8,33,132,80,74,74,169,115,16,66,8,33,132,16,74,41,37,165,212,57,8,33,148,16,66,8,41,165,148,66,8,33,132,16,66,8,33,149,146,82,8,33,132,16,66,40,165,148,84,82,10,33,132,16,66,8,165,132,82,82,74,41,133,
16,74,8,33,132,80,82,74,41,165,82,74,9,33,132,16,74,74,41,165,20,74,8,33,148,16,66,74,41,165,148,74,9,33,132,16,74,72,169,164,148,82,73,33,132,16,66,8,5,0,0,28,56,0,0,4,24,65,39,25,85,22,97,163,9,23,30,128,66,67,86,2,0,81,0,0,16,130,18,66,73,45,2,72,
41,38,173,134,72,57,39,173,214,18,57,164,28,197,26,34,166,148,147,150,66,6,153,82,76,74,9,45,116,140,73,75,41,182,18,58,72,169,230,28,83,8,41,0,0,0,130,0,128,0,19,64,96,128,160,224,11,33,32,198,0,0,4,33,50,67,36,20,86,193,2,131,50,104,112,152,7,0,15,
16,17,18,1,64,98,130,34,237,226,2,186,12,112,65,23,119,29,8,33,8,65,8,98,113,0,5,36,224,224,132,27,158,120,195,19,110,112,130,78,81,169,3,1,0,0,0,0,96,0,128,7,0,0,132,2,136,136,104,230,42,44,46,48,50,52,54,56,58,60,62,64,4,0,0,0,0,128,11,0,62,0,0,144,
16,32,34,162,153,171,176,184,192,200,208,216,224,232,240,248,0,9,0,0,4,16,0,0,0,0,0,1,4,32,32,32,0,0,0,0,0,16,0,0,0,32,32,79,103,103,83,0,0,64,83,0,0,0,0,0,0,139,173,239,94,2,0,0,0,10,149,235,126,24,50,53,53,196,209,205,199,200,193,199,195,197,190,195,
185,189,184,197,186,180,192,184,196,190,236,204,205,215,202,113,116,231,56,190,214,106,133,165,1,129,230,36,3,197,115,14,143,110,17,33,93,186,252,10,151,223,95,116,78,221,88,71,165,171,255,48,24,161,157,115,153,205,67,2,60,199,166,129,183,104,211,128,
107,211,98,1,77,168,69,194,72,144,224,104,180,73,104,154,155,159,155,155,223,252,187,177,56,172,172,12,58,231,48,140,131,101,89,93,90,45,107,32,163,206,165,0,43,84,77,185,137,133,89,185,137,95,213,10,19,84,50,45,169,17,35,18,196,130,128,48,112,206,81,
138,175,93,230,48,13,91,237,108,201,88,102,82,14,98,242,92,173,16,52,208,219,101,182,85,127,5,58,233,37,50,203,59,134,9,119,103,135,125,218,68,144,133,233,171,26,4,203,59,134,9,119,143,25,241,211,54,130,177,49,241,15,0,144,9,32,82,67,168,229,50,58,34,
38,44,65,134,33,100,143,4,16,81,142,200,0,150,35,161,90,52,124,71,68,13,219,145,202,45,67,50,85,117,78,163,209,64,77,149,70,59,80,205,2,48,119,2,208,30,0,255,160,29,204,158,15,130,251,85,12,95,60,184,130,20,83,80,23,31,56,51,138,180,166,132,243,168,42,
120,228,92,44,175,18,172,244,251,119,224,3,222,42,107,190,34,163,114,43,253,178,166,30,6,167,44,172,51,13,31,167,98,198,29,182,243,145,79,143,212,213,40,212,173,55,19,82,14,208,128,1,192,192,142,226,241,215,29,147,239,4,57,186,216,206,98,0,61,64,57,240,
173,127,28,94,249,165,97,154,184,188,218,124,223,207,103,158,211,121,87,83,38,139,232,235,90,168,177,229,242,104,227,190,238,39,226,110,242,160,154,1,44,2,191,3,0,0,0,120,18,108,203,86,169,53,169,211,35,251,179,3,0,49,149,0,64,57,161,0,0,0,148,188,212,
130,171,109,59,4,0,192,119,244,52,218,25,0,0,40,169,61,130,9,0,12,104,90,0,116,42,0,72,179,39,0,144,189,5,32,110,0,168,225,20,19,91,0,52,2,32,203,56,38,0,116,59,0,40,0,105,102,182,67,128,185,2,87,7,83,108,139,152,41,113,25,76,105,216,0,123,0,144,206,
156,6,17,154,234,15,98,33,9,246,192,122,238,147,77,27,154,152,185,189,31,129,214,134,132,2,19,16,187,228,108,90,129,92,108,216,34,175,42,68,131,76,125,169,37,104,114,223,62,183,110,65,158,165,125,40,116,251,211,1,62,185,149,89,221,250,69,43,35,253,232,
239,188,174,123,14,253,36,160,2,7,188,158,189,45,75,174,90,104,229,245,188,35,253,190,135,55,5,84,1,10,126,7,0,0,0,240,89,6,0,68,35,191,233,1,134,27,0,129,142,15,76,33,203,50,128,201,36,0,144,69,0,128,40,133,4,0,0,8,49,3,0,0,133,201,29,4,0,0,40,68,104,
130,23,2,0,224,27,161,57,99,67,0,0,192,114,207,164,105,1,136,93,142,0,192,222,237,54,0,208,225,206,34,0,88,111,246,22,0,186,13,0,194,212,10,64,220,11,0,235,1,224,98,5,0,186,13,192,0,14,24,134,179,128,70,141,179,10,13,143,107,0,66,154,243,25,216,11,9,
94,80,126,172,107,114,192,112,24,134,109,0,43,192,193,231,25,195,62,226,110,146,78,205,213,197,0,37,67,8,28,0,48,88,156,1,30,169,101,97,105,244,34,181,49,238,244,110,170,126,166,160,48,1,89,146,180,154,141,157,34,46,67,105,227,156,207,230,211,207,28,
162,129,88,146,248,37,66,2,80,79,0,0,159,53,0,16,77,184,3,194,29,144,182,77,33,135,35,205,39,195,26,36,147,109,110,9,0,128,66,1,0,50,138,4,0,0,144,33,0,0,144,178,16,0,0,0,183,128,31,3,0,0,36,170,15,105,6,0,157,144,23,0,216,51,206,2,160,238,0,128,78,30,
173,8,0,86,135,0,64,76,149,5,0,86,11,156,13,64,231,24,0,160,18,112,192,0,120,0,136,3,16,8,177,147,182,161,146,138,109,67,65,204,71,206,239,3,204,3,156,13,103,131,72,198,74,10,18,208,6,87,0,18,4,95,71,6,46,14,95,60,112,56,172,184,211,88,188,179,222,198,
58,2,192,1,30,169,101,65,153,137,109,73,99,158,125,135,223,77,10,223,148,200,24,90,205,38,72,173,94,166,50,231,181,118,132,221,48,108,83,2,99,240,59,0,0,0,128,207,72,0,152,152,236,216,94,96,1,203,166,199,32,197,135,104,236,6,24,192,16,134,84,0,64,13,
0,160,97,82,1,0,192,188,14,0,96,48,73,201,0,0,0,10,48,70,130,48,64,198,70,83,121,138,2,0,216,37,198,103,172,2,0,0,78,231,137,73,41,0,34,157,74,1,232,112,231,227,0,96,108,114,10,0,198,119,30,0,208,221,22,0,12,50,161,0,208,237,8,0,56,0,192,10,224,4,14,
66,2,22,27,154,208,189,84,208,138,110,1,40,40,3,173,25,64,140,1,136,88,148,157,33,176,15,128,3,32,81,32,225,112,64,8,6,78,134,11,100,153,105,90,18,74,192,37,3,254,168,101,161,111,124,211,82,155,87,223,121,86,231,157,83,102,99,104,53,27,37,22,226,162,
133,49,239,250,136,196,253,30,116,202,144,51,242,15,0,80,2,128,38,211,231,20,0,160,33,60,128,49,29,155,194,48,197,135,48,100,22,140,225,244,24,0,48,195,142,0,0,32,66,61,54,0,0,162,84,4,0,0,20,184,147,52,241,80,0,128,200,86,207,212,1,160,49,53,0,48,154,
42,10,0,123,247,6,0,236,13,0,52,147,155,15,0,114,6,164,0,96,208,52,0,32,86,129,246,0,48,140,5,16,192,193,0,97,0,140,161,64,119,11,106,136,8,217,6,128,1,176,199,52,128,122,6,104,15,140,113,128,128,48,248,224,109,67,104,20,26,88,43,124,44,144,7,24,128,
116,56,128,205,203,189,179,81,67,205,93,81,46,1,62,185,101,121,63,197,182,148,121,174,181,99,171,239,87,80,155,9,219,1,180,106,150,65,108,184,40,233,121,206,107,70,216,13,131,98,38,224,217,47,81,68,2,200,118,35,0,128,239,106,3,192,88,37,128,181,77,219,
36,200,77,143,41,134,45,24,195,185,235,24,5,0,128,50,45,21,0,248,168,181,21,0,0,214,85,166,74,0,0,0,210,180,243,226,0,0,208,110,251,38,10,64,196,212,38,2,128,205,56,6,0,86,13,0,96,111,2,0,49,105,7,0,216,172,12,4,128,61,115,2,128,174,4,160,0,196,51,0,
80,1,16,64,32,9,28,0,124,134,8,152,195,24,241,173,27,32,2,160,246,82,4,112,25,64,52,14,71,27,33,137,76,94,170,77,49,173,40,64,156,194,4,36,39,4,128,3,72,17,103,142,188,252,62,245,24,129,13,64,2,190,57,150,97,153,196,197,84,230,184,159,45,252,123,56,178,
13,20,60,172,166,181,157,26,113,177,210,156,247,115,68,124,79,193,137,172,46,221,48,128,22,0,128,207,38,0,192,110,234,1,164,109,219,20,198,233,145,81,172,122,72,192,48,166,191,97,89,228,12,0,64,13,179,85,0,80,165,93,0,0,224,178,32,0,0,176,37,91,33,34,
12,108,97,41,218,110,19,0,0,165,99,164,219,137,0,96,16,0,144,228,141,3,128,209,148,179,0,192,110,0,0,145,55,14,0,206,203,30,0,96,48,13,0,232,106,0,30,36,224,56,87,128,6,20,56,96,0,0,39,29,64,186,63,146,194,102,26,0,3,8,93,85,8,134,51,0,28,96,1,180,177,
40,64,51,164,40,0,168,182,24,243,0,49,192,78,9,140,200,56,135,82,160,1,120,0,190,89,166,217,58,177,93,225,236,227,154,121,212,253,6,187,36,6,15,163,105,17,150,150,237,20,247,56,175,25,225,253,14,57,50,121,240,59,0,0,0,128,207,158,0,80,140,123,0,97,187,
128,165,162,78,143,140,138,45,195,117,0,32,138,4,0,26,134,35,0,0,68,249,54,0,0,200,110,0,0,0,172,0,227,110,56,20,0,192,119,200,112,57,81,0,0,8,63,111,26,147,3,0,98,80,0,168,169,183,0,144,12,24,5,0,230,30,0,96,144,17,0,216,223,17,0,60,24,96,0,33,86,10,
180,38,0,16,64,224,81,104,1,224,5,160,176,175,41,48,225,6,144,3,138,172,130,64,4,242,139,192,57,168,4,57,203,68,144,200,137,239,96,13,8,61,41,64,128,134,39,16,82,130,0,13,9,69,53,61,79,89,178,16,15,148,244,0,94,106,22,86,47,125,179,90,59,59,59,47,116,
31,103,81,246,148,154,150,118,43,117,179,218,216,231,185,243,194,247,43,44,145,237,201,63,0,192,2,0,147,234,190,83,0,128,122,17,192,56,166,163,150,138,92,10,25,181,90,24,78,97,0,64,131,69,0,0,64,253,243,21,13,0,128,73,163,5,0,0,64,210,41,226,98,3,0,224,
150,19,1,128,24,152,15,0,198,51,121,0,96,92,61,3,0,43,0,16,150,73,1,160,49,48,11,0,53,88,5,128,91,128,3,72,100,7,192,24,0,0,26,160,40,156,0,74,65,117,25,48,193,240,44,160,1,48,199,4,30,17,8,9,206,1,129,136,102,114,26,228,67,120,52,87,3,112,138,18,66,
168,0,131,21,138,178,217,1,9,58,192,230,179,125,144,44,51,20,8,52,94,122,166,153,49,45,95,105,140,203,28,121,84,167,76,27,136,199,84,52,11,198,180,97,165,214,46,107,228,89,247,28,180,1,60,248,29,0,0,0,192,111,0,0,142,99,59,20,206,34,67,180,5,62,11,0,
100,17,0,32,74,9,0,84,147,12,73,0,0,124,139,55,6,69,0,0,192,74,87,180,136,11,0,0,160,197,50,106,22,0,178,3,21,0,105,102,0,0,152,13,72,3,0,58,104,20,0,50,121,2,0,166,86,0,48,204,5,0,34,175,89,0,96,169,2,6,20,100,95,2,90,160,1,7,128,46,3,30,96,186,112,
49,199,38,235,2,32,205,216,0,141,64,231,161,8,224,46,154,166,77,196,128,146,136,238,253,37,4,16,104,12,0,87,60,201,206,36,176,76,89,171,33,57,103,34,4,203,236,59,249,25,135,2,13,7,190,122,22,118,223,124,179,204,185,53,99,58,247,43,144,146,204,158,87,
201,50,168,187,111,147,113,142,107,199,116,238,27,72,73,102,207,110,104,64,2,0,248,60,6,0,69,205,29,176,166,99,35,2,48,61,50,68,91,224,83,177,162,2,0,32,171,149,0,64,70,26,4,0,0,26,57,0,0,128,228,226,215,98,1,0,128,36,244,203,38,0,144,25,92,1,48,172,
46,0,96,239,121,0,0,195,8,0,33,47,0,64,228,165,0,160,131,68,0,128,125,0,208,88,0,14,38,129,182,1,120,130,86,33,198,2,166,136,22,0,58,199,16,16,3,252,224,0,76,104,132,76,26,146,124,105,104,204,139,8,1,165,236,64,1,90,136,29,32,1,124,195,16,202,32,80,176,
80,73,151,26,147,0,12,0,210,158,138,102,193,220,198,216,202,104,135,26,205,167,222,176,24,32,82,46,61,179,96,106,219,184,82,41,235,165,249,186,231,184,205,0,30,253,146,2,128,44,155,2,0,248,172,36,0,76,78,199,228,183,17,129,140,94,35,138,45,113,136,9,
174,20,11,0,32,52,10,0,144,181,6,1,0,96,166,226,162,2,0,0,118,218,200,83,3,5,181,56,23,0,218,65,115,0,64,58,121,12,0,44,43,1,0,251,6,0,208,149,17,0,162,121,1,128,202,93,54,1,0,199,253,64,65,237,93,67,2,32,0,240,3,156,0,208,136,54,52,101,21,141,18,33,
103,2,36,0,237,50,180,240,112,128,3,99,1,201,120,197,78,161,141,54,2,0,230,32,145,64,210,153,220,144,194,161,23,9,76,150,223,101,251,143,192,1,1,0,190,138,166,214,94,122,94,218,28,183,94,98,127,119,195,64,27,104,132,116,90,102,214,94,250,80,202,108,151,
42,177,63,249,48,109,32,17,179,127,0,128,72,0,168,52,250,60,73,0,40,170,238,128,176,77,59,198,233,249,32,136,25,196,17,224,16,0,48,243,212,101,0,0,34,249,34,0,0,144,67,11,0,0,32,39,141,237,176,211,8,0,64,26,221,123,217,20,0,114,22,1,64,186,114,8,0,232,
32,17,0,116,101,0,0,181,18,0,232,110,8,0,58,120,154,45,0,122,24,1,4,246,163,128,0,13,196,172,208,59,128,117,24,10,213,140,41,77,2,181,3,88,204,60,27,17,12,96,0,43,209,254,186,5,25,3,128,169,168,129,246,225,126,18,136,195,128,164,64,72,150,173,34,0,20,
222,106,118,246,84,248,69,72,163,29,178,229,81,247,13,118,201,3,43,64,169,216,216,99,124,235,194,104,203,22,147,190,175,96,217,160,80,145,0,126,7,0,0,0,240,173,3,128,152,18,144,30,64,218,182,90,14,99,102,50,106,54,131,152,130,0,128,154,0,32,138,36,24,
0,0,194,152,243,1,0,0,64,210,14,0,0,0,83,76,229,192,53,53,0,0,206,8,145,115,170,0,0,144,80,119,232,1,0,209,129,0,32,233,82,0,48,236,63,2,128,89,94,1,192,234,8,0,228,229,2,128,91,0,112,2,24,188,16,112,144,28,0,120,160,173,0,24,3,1,161,203,144,34,5,235,
1,20,2,64,132,49,20,147,5,238,48,14,96,53,100,173,59,24,14,5,0,117,10,96,0,240,16,142,149,225,209,64,59,28,190,165,61,25,82,1,0,18,190,138,22,214,220,214,80,90,115,234,173,169,186,239,220,102,36,75,66,40,88,88,83,91,67,40,195,161,143,166,210,253,78,49,
35,89,1,253,18,25,0,40,78,0,0,124,22,1,0,245,116,7,132,99,170,37,96,14,35,88,31,1,50,2,65,65,0,0,163,207,21,0,100,9,3,0,128,209,48,0,0,32,146,46,167,198,4,0,0,171,248,200,96,10,0,131,140,3,64,231,50,22,0,186,124,20,0,186,52,10,0,251,6,0,96,117,0,0,49,
152,6,0,60,11,176,2,128,3,156,109,14,32,216,184,6,6,26,43,136,186,159,36,239,22,179,152,113,46,233,199,1,6,112,0,163,97,10,150,168,100,0,52,64,211,93,0,16,210,33,101,227,125,67,58,129,0,26,27,96,43,157,122,91,102,192,1,222,90,54,246,210,99,48,105,164,
107,182,188,160,251,61,244,13,96,69,220,252,50,107,41,53,119,161,133,107,142,136,243,201,171,6,168,68,4,191,32,1,40,106,0,0,62,27,0,64,165,234,216,244,118,12,25,61,31,4,49,59,2,28,198,162,8,0,64,232,115,5,0,153,75,5,0,0,25,13,197,31,0,0,64,21,31,23,10,
64,13,154,1,0,49,48,31,0,194,10,2,0,44,107,17,0,152,11,0,184,29,0,200,106,33,32,144,223,12,0,24,91,64,20,84,172,128,16,28,0,156,22,144,26,128,15,84,128,229,164,72,10,50,18,134,224,76,4,86,76,50,236,128,143,29,107,130,1,0,9,32,87,192,160,0,78,185,215,
20,160,60,107,75,92,2,238,109,203,24,56,0,222,90,214,214,210,244,209,149,25,143,57,194,239,126,138,54,162,173,4,157,94,105,47,101,109,66,218,241,168,22,254,125,15,117,9,37,43,130,223,1,0,0,0,252,6,0,96,171,165,183,135,145,70,150,65,134,49,35,56,38,1,
160,148,0,64,25,0,48,87,43,138,42,0,128,163,223,223,10,10,0,0,206,66,123,13,59,1,0,64,19,244,53,163,25,0,44,90,118,14,0,22,45,3,0,178,172,2,64,88,65,1,224,189,0,224,108,1,192,178,4,0,12,204,9,0,112,2,136,198,190,60,192,193,145,0,122,90,74,99,1,4,222,
34,16,21,141,165,17,17,219,9,110,7,136,209,190,223,61,96,156,0,7,112,180,219,174,59,19,4,232,163,16,130,2,148,19,24,243,147,19,0,120,107,128,169,132,97,155,90,112,1,56,190,138,54,252,156,216,182,17,235,122,107,190,187,51,160,152,1,89,2,43,191,210,158,
219,184,104,37,229,169,229,49,55,63,164,146,11,150,68,255,0,0,3,0,168,212,125,27,1,192,145,0,198,177,77,155,128,233,195,117,28,98,2,0,64,184,123,151,0,0,128,88,154,213,6,0,0,38,146,6,0,0,16,121,23,141,185,41,0,192,226,150,191,2,0,150,93,28,0,96,122,25,
0,104,156,54,7,128,100,156,0,96,189,20,0,44,2,128,54,119,73,4,0,78,0,1,66,196,9,56,43,28,112,54,180,7,192,248,73,27,169,198,234,152,102,192,156,192,145,225,113,25,18,16,160,35,64,186,166,67,100,72,178,20,144,128,0,2,33,11,16,219,99,55,126,105,80,43,96,
3,101,236,209,194,3,158,122,150,225,210,172,77,41,45,31,107,228,89,221,71,40,37,25,69,204,141,91,26,206,179,182,73,169,228,163,143,60,210,125,15,18,169,17,179,223,1,0,0,0,124,86,3,0,138,211,220,1,97,242,171,37,112,122,100,100,13,49,153,0,0,32,139,0,0,
81,76,54,0,0,56,22,1,0,0,120,84,51,194,34,132,41,0,128,85,108,251,84,34,0,0,144,48,111,110,83,0,200,230,3,64,78,195,4,128,238,63,7,0,140,0,32,135,97,1,0,150,2,68,64,87,56,44,0,227,1,60,80,112,190,82,0,14,72,192,35,48,25,0,28,35,84,194,48,238,87,229,149,
98,1,12,16,225,154,65,199,0,109,80,128,16,180,157,75,230,160,202,62,128,217,249,13,16,40,229,160,184,68,10,192,146,220,128,55,201,18,0,58,25,167,17,0,126,74,54,86,107,219,198,141,22,143,107,132,127,151,67,74,9,21,17,225,176,74,131,62,181,193,148,145,
142,167,133,189,231,112,74,4,92,68,32,191,100,100,2,240,76,2,0,248,13,0,192,49,76,181,20,152,99,8,48,214,16,83,199,78,233,6,0,96,78,230,0,160,70,0,0,243,59,185,24,0,0,234,223,90,202,233,12,0,0,32,222,114,205,102,26,0,105,172,64,0,96,209,146,108,0,192,
181,211,2,64,246,79,0,80,108,2,0,150,125,1,128,90,1,0,11,203,230,0,192,91,64,11,0,198,192,35,149,9,135,11,9,40,0,240,12,48,240,156,163,248,34,94,0,20,216,59,187,96,146,99,0,104,230,57,109,104,99,114,0,2,7,192,76,235,12,81,1,10,29,82,182,73,11,14,192,
2,155,10,40,0,79,103,103,83,0,0,64,171,0,0,0,0,0,0,139,173,239,94,3,0,0,0,74,3,224,72,22,181,191,187,193,192,191,192,185,190,188,197,197,197,201,193,195,200,211,208,204,201,212,158,58,214,234,88,214,24,140,120,232,34,209,185,215,41,3,131,83,26,244,54,
70,23,135,237,41,194,251,6,165,36,64,131,223,1,0,0,0,124,14,9,0,121,202,182,109,43,129,19,100,36,102,24,83,113,12,0,168,1,0,148,219,10,0,0,22,45,150,166,7,0,192,25,249,140,219,76,1,0,128,176,237,223,141,57,0,98,73,46,0,152,45,155,22,0,246,237,47,0,68,
151,42,0,186,63,0,32,78,175,0,72,246,3,128,249,10,18,0,208,22,2,32,128,204,28,72,0,65,0,133,68,83,128,251,167,154,225,33,140,20,246,24,0,2,216,233,16,32,24,38,129,141,129,125,175,97,166,243,1,104,5,192,25,58,199,0,109,5,160,131,95,35,181,186,100,18,230,
197,6,135,2,94,74,150,97,159,34,15,105,183,213,136,73,223,199,113,53,3,202,2,22,173,52,51,53,109,112,105,148,147,70,83,125,15,67,191,100,133,44,192,63,0,64,53,1,160,54,213,247,36,0,160,168,1,210,177,77,155,194,171,48,89,5,63,32,0,96,173,233,6,0,128,210,
36,111,0,0,0,158,83,7,0,0,96,90,44,61,72,163,0,0,56,231,222,0,0,178,203,79,3,128,56,173,5,64,196,233,21,0,57,111,0,192,176,210,0,64,71,64,128,105,167,129,0,236,43,208,66,0,206,65,224,32,193,97,4,18,144,18,128,83,3,24,246,152,152,224,29,100,129,57,5,184,
147,137,19,194,0,64,0,25,22,102,211,50,166,49,192,1,241,6,8,64,132,155,173,211,116,9,38,153,252,103,77,113,3,32,68,76,212,4,0,158,74,118,65,47,106,155,210,40,199,158,242,66,247,13,171,54,80,52,122,204,178,112,44,107,76,90,58,122,202,179,238,43,184,75,
54,225,134,22,244,0,0,188,1,244,1,225,216,166,90,10,39,38,70,16,107,136,147,231,194,104,178,0,0,124,195,164,2,128,44,34,49,1,0,96,18,227,211,6,0,0,81,165,115,3,10,0,0,9,115,103,0,32,186,130,12,0,72,151,206,0,128,7,83,0,176,40,0,186,178,5,0,246,42,4,160,
131,68,0,192,15,180,21,45,84,199,0,84,179,106,64,64,3,0,253,76,80,144,165,128,160,186,34,226,155,28,64,248,68,14,7,248,216,85,160,104,172,168,162,253,40,2,108,0,34,48,151,255,232,83,3,99,146,155,184,205,50,37,7,64,228,165,132,56,56,20,0,126,74,54,193,
212,248,38,164,217,14,61,196,164,239,119,172,109,168,56,33,52,185,21,225,52,197,198,165,221,142,30,98,171,187,97,216,150,172,40,33,228,151,49,9,128,226,24,0,192,103,3,0,168,148,142,105,171,165,224,102,42,214,16,39,140,124,168,3,0,208,61,87,0,144,201,
16,0,0,248,183,211,36,9,0,0,154,231,58,216,2,192,10,99,58,0,96,233,178,0,16,93,38,1,0,179,43,0,96,46,1,0,153,37,5,0,203,6,128,210,230,1,0,121,192,2,66,196,220,15,96,72,144,160,161,12,156,128,64,186,20,65,37,206,25,27,17,211,36,0,18,160,45,99,3,72,2,70,
105,64,105,140,105,202,114,50,0,7,2,149,255,41,165,30,20,142,134,248,73,58,154,8,112,127,108,96,163,130,244,15,160,208,32,0,190,122,246,252,212,232,69,139,39,156,86,8,255,126,199,82,162,74,54,193,198,26,219,216,150,52,29,122,104,42,220,55,204,37,26,227,
119,0,0,0,0,159,153,0,16,13,182,109,170,69,111,98,42,214,16,114,8,14,0,40,18,0,184,187,249,6,0,0,194,20,12,60,28,2,0,96,53,60,206,103,20,0,0,234,230,234,206,19,0,200,44,63,42,0,228,45,41,0,216,191,31,0,194,146,57,0,56,142,1,0,43,38,0,216,223,0,8,154,
46,1,0,167,1,17,12,73,20,218,34,65,20,104,1,8,9,64,27,4,0,93,66,161,85,230,10,181,154,10,112,128,182,186,12,1,4,214,76,201,0,16,100,69,137,164,57,14,225,0,128,187,131,210,80,174,19,196,181,33,185,243,47,53,168,15,134,115,183,182,19,39,8,0,158,122,246,
246,88,206,77,27,41,31,167,60,234,190,199,186,100,198,116,134,141,61,54,107,108,37,228,165,60,250,126,15,167,100,69,248,7,0,8,0,80,169,251,105,4,0,141,106,0,64,225,216,106,109,26,220,196,84,172,33,228,200,24,0,48,211,17,0,0,16,221,140,104,0,0,80,203,
13,0,0,32,158,91,92,210,8,0,64,106,173,138,0,128,129,185,1,0,77,150,8,0,196,210,20,0,100,44,128,160,179,252,17,0,188,1,4,136,197,121,1,0,29,1,98,64,5,109,158,64,44,144,112,160,13,40,12,4,160,116,65,0,146,213,36,86,76,2,13,16,145,253,37,170,9,24,11,144,
64,104,134,44,173,38,190,156,67,3,16,192,128,31,6,198,97,23,224,198,116,51,19,6,212,83,74,65,16,235,110,18,5,0,158,106,142,65,109,218,102,181,150,183,90,243,213,253,1,98,66,112,28,42,246,97,111,218,166,141,17,15,85,242,232,251,61,136,13,148,240,15,0,
80,3,128,40,154,240,225,0,0,30,64,216,166,90,170,72,143,169,88,199,4,226,24,0,80,202,16,0,0,68,141,10,0,212,8,0,0,32,42,156,97,81,1,0,192,81,59,77,253,152,2,0,128,29,214,255,58,1,96,117,53,5,0,233,178,0,160,153,37,0,192,42,0,176,156,71,0,136,17,0,176,
28,1,112,10,136,161,136,166,205,8,48,212,228,9,36,11,141,3,216,6,35,32,89,12,24,87,164,4,233,138,64,16,104,5,36,218,69,37,115,4,26,27,0,109,117,24,67,98,145,198,0,36,32,17,204,213,136,120,41,129,20,217,249,184,165,200,81,163,243,28,14,0,126,106,246,106,
43,24,39,211,113,142,72,221,55,72,145,193,101,232,216,91,45,58,46,219,49,75,36,186,225,33,71,38,252,146,20,0,148,53,0,0,223,30,0,48,46,129,112,7,164,169,50,41,144,113,152,196,28,132,201,131,201,5,0,16,238,87,0,0,13,46,1,0,128,180,150,36,5,0,0,144,216,
104,0,0,192,14,106,189,24,0,0,136,75,101,24,0,204,172,0,64,182,195,4,0,59,1,128,238,95,33,0,194,1,8,93,172,20,0,58,6,224,21,128,116,53,2,100,225,35,0,92,6,188,0,193,120,129,152,222,76,249,37,1,19,0,207,103,41,96,37,97,78,49,48,241,150,243,43,100,227,
1,62,56,34,0,210,11,94,0,52,41,116,222,195,16,114,218,128,222,177,70,65,21,18,0,94,90,118,225,216,196,182,148,59,157,103,138,240,85,82,48,149,25,29,251,112,44,125,91,218,73,231,153,34,124,21,56,166,50,126,68,36,0,99,245,73,0,160,177,234,123,156,0,112,
82,3,164,227,6,192,176,109,10,38,244,200,200,42,248,27,11,0,128,120,47,207,2,0,192,30,99,119,0,0,176,167,59,40,1,0,192,86,32,0,0,128,16,10,0,217,65,18,0,208,1,0,224,51,0,136,129,45,0,88,0,32,98,144,6,0,100,108,0,10,64,134,37,66,175,145,85,59,100,208,
144,103,0,162,1,128,183,19,20,116,71,211,76,25,110,0,135,2,211,14,225,14,72,2,31,32,1,52,99,131,68,196,91,6,192,7,5,32,0,194,242,166,15,129,146,47,155,173,89,54,181,130,73,232,74,216,33,184,19,64,1,94,74,246,246,60,19,155,54,70,186,118,225,223,239,225,
175,72,41,57,90,243,164,155,54,90,186,118,225,119,127,194,190,34,126,7,0,0,0,240,89,38,0,84,38,59,22,91,45,109,18,19,134,19,60,23,3,0,202,0,64,212,35,0,0,16,217,237,226,240,54,21,0,192,53,115,166,233,77,0,0,16,59,25,126,124,4,64,116,106,25,0,144,12,146,
5,0,145,215,0,128,188,4,144,7,128,72,6,20,0,88,17,0,204,166,202,2,64,188,0,74,17,68,76,42,160,129,6,49,78,184,230,242,49,176,240,6,20,96,245,171,28,189,246,3,22,208,140,115,206,160,233,30,112,71,226,81,53,31,72,99,32,0,40,99,255,174,164,6,129,152,210,
4,116,170,175,200,28,136,12,198,1,192,33,188,135,163,47,28,30,58,246,225,212,198,166,165,17,174,46,143,234,62,192,53,101,51,74,142,153,169,105,219,86,134,235,146,103,117,31,193,85,9,191,3,0,0,0,248,13,0,192,86,217,38,173,96,4,25,97,56,9,33,55,0,16,53,
1,64,148,18,0,40,28,206,184,155,2,0,216,233,114,18,80,0,0,196,8,164,75,237,2,0,0,104,154,198,102,0,16,249,83,40,0,50,83,31,47,0,172,140,5,128,238,25,0,192,9,0,228,71,0,0,11,2,96,102,5,0,56,3,80,128,208,73,1,72,24,18,0,2,106,155,69,122,12,219,128,48,19,
18,37,24,205,157,24,75,124,99,208,128,69,119,32,164,248,108,2,34,192,152,1,23,20,35,66,0,134,236,212,157,133,7,72,171,106,143,46,243,211,143,49,149,98,222,17,2,165,38,93,120,78,2,216,65,0,254,9,54,246,218,125,52,169,133,107,137,94,254,115,133,209,0,148,
134,189,181,118,31,77,106,174,37,122,203,171,48,152,6,192,13,5,8,0,0,223,134,0,128,108,10,24,91,216,54,197,77,24,97,19,60,119,72,177,24,0,128,4,149,1,64,198,137,2,0,0,215,168,40,0,0,96,85,197,213,4,0,0,195,8,39,0,208,236,128,0,192,48,119,0,160,80,21,
129,8,0,26,249,16,0,25,11,0,34,31,0,200,8,96,3,161,6,171,5,192,184,200,152,125,195,142,245,122,108,245,44,205,164,197,77,185,92,198,145,91,45,67,255,80,117,113,23,67,39,104,27,131,121,131,1,226,156,161,171,163,1,7,24,56,115,215,69,100,48,6,192,172,63,
219,224,128,190,58,92,2,178,202,220,104,61,0,220,207,195,192,152,5,82,214,209,6,144,6,62,42,78,214,210,136,177,164,153,206,46,212,110,56,220,102,202,156,138,163,53,181,12,46,204,116,172,22,126,55,12,106,51,209,145,31,145,9,64,212,78,3,0,217,232,219,4,
0,40,202,4,184,3,150,206,84,33,125,250,25,158,77,86,146,102,6,0,144,175,217,17,0,0,57,43,105,9,0,0,48,218,144,24,26,0,0,8,163,49,0,0,0,43,137,22,0,172,0,0,119,0,64,61,0,154,174,4,0,100,41,0,56,102,20,0,245,2,0,14,0,144,15,120,241,115,62,23,209,154,11,
183,83,165,158,113,79,44,135,153,253,14,21,5,131,211,224,199,190,132,26,0,191,33,124,245,1,238,158,32,31,71,105,24,144,128,179,254,0,92,193,205,39,1,28,210,96,135,3,95,142,58,160,113,21,20,44,156,186,68,156,43,118,24,161,177,6,16,254,25,142,193,84,138,
177,165,217,86,169,169,116,191,194,219,70,128,165,167,229,20,140,17,155,149,70,57,244,212,124,186,111,88,219,40,144,212,15,25,0,68,67,35,0,100,54,245,211,0,0,162,168,3,0,37,191,98,115,20,155,202,88,146,77,14,33,230,6,0,8,89,90,29,1,0,32,132,127,43,0,
0,128,18,192,0,0,96,231,3,0,24,86,243,1,128,156,10,0,92,28,0,128,165,0,8,157,52,0,64,179,24,152,97,207,14,0,186,20,166,0,128,113,177,129,128,51,4,49,22,92,198,150,139,87,72,83,109,202,1,1,65,129,213,124,70,188,0,36,0,140,23,6,179,38,65,215,231,57,211,
157,166,31,7,0,142,48,67,225,212,3,160,177,247,119,194,140,161,1,8,210,36,160,6,250,49,222,61,6,152,56,134,242,51,129,224,183,104,227,8,52,190,9,54,234,210,154,161,216,237,228,66,253,185,199,109,70,166,180,236,236,181,145,67,73,167,29,92,243,253,115,
87,104,38,132,223,1,0,0,0,124,206,1,64,212,108,149,173,150,54,33,140,99,199,56,53,0,160,72,0,160,60,69,1,0,64,200,138,164,120,4,5,0,192,42,88,66,0,5,0,128,240,173,79,13,154,0,64,50,85,182,0,144,201,155,1,128,185,11,0,24,23,0,224,42,0,68,243,3,0,237,8,
0,12,30,0,192,1,60,4,8,232,228,0,94,26,118,28,25,194,229,76,250,114,175,65,218,12,77,196,38,32,190,218,75,3,208,224,0,152,219,192,32,90,2,193,120,80,32,124,202,6,0,132,30,242,214,128,129,184,193,46,96,192,176,91,145,40,31,233,0,8,99,35,169,95,60,249,
20,170,181,13,82,6,158,233,141,246,82,138,139,84,70,60,118,205,87,55,28,106,19,1,74,201,201,154,26,46,82,27,241,188,228,81,221,112,184,77,4,225,119,0,0,0,0,159,19,0,160,90,115,180,105,80,73,234,225,19,194,0,0,141,3,0,81,206,13,0,0,144,163,102,164,168,
0,0,68,200,254,78,39,8,0,0,102,224,222,212,0,64,50,72,26,0,180,131,10,0,48,41,0,160,93,2,108,0,192,92,61,0,160,237,88,0,48,14,5,64,146,22,160,0,36,227,48,3,40,99,115,184,175,165,251,54,129,242,215,117,138,108,83,137,117,250,239,1,72,178,12,205,10,192,
215,208,43,129,197,38,249,109,7,192,62,32,16,41,139,52,64,234,96,31,169,129,100,48,32,106,72,10,180,79,1,179,109,21,174,23,46,113,123,48,145,103,219,23,68,167,3,30,185,53,153,101,38,182,169,237,52,215,84,250,121,5,194,68,153,210,113,204,172,141,110,211,
56,105,41,47,244,248,21,155,40,225,119,0,0,0,0,159,101,0,64,81,183,57,211,166,78,210,215,9,59,73,2,32,139,0,0,81,106,82,0,0,80,113,86,95,35,1,0,128,149,224,99,198,34,0,0,136,191,245,196,89,1,144,49,16,0,12,76,198,0,160,89,33,0,28,15,0,13,0,48,172,52,
0,176,82,0,48,28,188,1,128,140,0,167,0,181,87,10,12,223,133,187,175,244,111,226,34,196,72,54,27,205,129,0,113,130,65,197,10,31,156,193,25,6,132,108,71,10,228,37,128,211,70,101,219,16,72,128,59,28,42,73,172,96,8,224,189,206,197,128,22,32,65,56,231,50,
28,134,88,127,149,42,152,209,135,164,4,4,240,50,61,251,166,73,81,0,222,168,141,217,125,230,151,161,236,120,71,77,235,126,195,190,137,16,163,227,18,172,147,184,116,109,187,186,230,211,253,142,125,3,193,79,81,10,0,198,234,147,0,192,20,99,190,21,0,80,173,
1,210,209,42,21,149,16,66,102,179,9,57,164,68,0,0,110,119,183,0,0,112,111,95,106,0,0,176,215,174,0,0,0,76,26,20,0,178,83,70,0,160,205,19,0,116,143,0,30,0,146,189,5,192,249,27,160,24,199,9,0,183,128,86,0,86,142,5,60,234,200,254,200,149,77,156,43,203,36,
103,94,78,152,220,158,8,187,154,160,142,153,144,114,214,10,251,130,166,119,171,103,248,123,247,144,175,33,205,200,199,152,171,22,19,143,148,208,100,42,164,77,117,70,211,113,167,108,18,197,227,82,201,130,112,39,4,140,14,86,163,99,188,85,54,164,40,124,
143,13,236,125,13,66,219,142,133,167,6,197,0,126,136,245,225,58,113,205,133,145,175,18,234,253,30,124,19,152,206,177,15,151,73,92,115,97,196,179,11,245,23,7,160,137,20,191,3,0,0,0,248,56,36,0,152,194,29,176,194,212,36,35,115,236,201,25,0,160,6,0,80,71,
49,0,0,196,151,230,13,0,0,224,226,50,13,111,169,31,0,192,214,14,209,138,0,0,32,206,250,83,158,10,128,142,123,91,0,176,10,0,82,172,129,41,0,144,85,1,128,44,5,128,227,216,169,240,10,248,1,130,122,92,164,178,69,2,0,209,244,185,42,141,55,157,198,11,152,251,
72,75,3,29,108,52,226,142,96,97,141,253,206,116,49,190,167,71,215,137,217,225,48,240,149,33,182,36,123,96,34,226,0,67,24,91,119,48,219,16,99,2,36,5,128,33,249,151,140,172,136,58,203,167,183,50,159,220,39,109,12,88,164,137,49,249,199,6,225,88,6,30,88,
141,217,117,38,175,59,59,159,43,52,173,222,225,182,17,150,30,71,199,197,94,91,174,185,180,203,209,67,179,116,207,176,181,81,149,129,146,127,0,128,241,14,0,49,133,207,81,1,0,38,220,43,128,73,79,79,196,44,209,79,88,60,1,128,201,184,10,0,0,212,145,140,20,
0,0,112,8,207,203,0,0,128,10,74,85,79,19,0,0,179,106,147,100,202,0,64,117,79,0,160,182,1,128,110,215,0,64,11,0,187,2,0,212,5,0,142,123,18,0,136,6,78,1,65,180,133,0,128,100,88,101,102,78,52,24,28,16,227,248,30,4,134,53,180,53,255,24,168,233,192,201,180,
29,218,111,199,101,28,34,42,52,205,72,184,218,201,51,227,3,129,15,6,115,9,104,208,141,193,56,189,115,96,1,204,156,53,167,204,88,156,116,175,66,98,165,60,73,209,84,204,25,2,30,72,141,234,81,200,49,229,217,142,94,34,254,185,194,104,59,84,6,164,82,112,179,
167,34,46,76,108,241,172,18,147,185,231,240,183,209,226,160,228,23,178,0,64,29,0,192,47,77,94,0,64,108,0,0,199,230,55,185,235,196,242,53,188,146,88,0,0,163,79,77,0,32,255,21,0,0,128,101,106,2,0,74,189,169,13,0,0,96,12,125,85,175,31,0,0,192,156,116,5,
0,176,226,100,0,192,14,5,128,157,0,0,91,0,69,76,2,0,97,15,0,32,43,0,0,7,0,206,110,16,196,7,78,129,116,113,220,33,7,32,45,200,30,105,0,88,2,13,149,206,10,207,154,0,222,4,132,144,79,85,16,246,173,60,10,97,103,192,21,67,134,214,157,19,146,58,11,65,148,44,
128,0,98,208,228,28,34,45,246,11,128,26,169,248,125,160,96,134,157,81,0,222,87,157,248,199,89,153,233,82,186,248,204,179,122,124,214,174,11,132,66,197,73,217,75,121,25,204,114,204,210,171,4,247,43,248,219,176,209,24,220,16,176,0,0,126,3,0,224,231,85,
76,84,172,19,213,255,139,215,181,41,0,0,124,91,204,122,0,96,89,159,0,160,15,117,175,54,0,0,64,108,99,131,11,10,0,0,64,182,236,247,131,157,207,0,16,79,221,177,4,0,245,211,103,5,128,77,93,13,0,174,0,0,162,30,0,246,225,53,0,104,239,20,0,240,62,15,0,32,30,
78,161,5,225,160,92,13,24,244,58,237,180,67,100,78,10,21,16,23,192,201,176,40,161,104,112,184,141,195,0,107,255,119,239,122,107,0,195,184,68,58,16,148,118,52,132,132,56,243,235,8,148,180,14,61,145,141,194,26,154,243,67,164,204,71,250,90,193,39,131,23,
224,0,204,156,62,209,8,144,232,60,128,3,79,103,103,83,0,4,83,234,0,0,0,0,0,0,139,173,239,94,4,0,0,0,39,227,103,179,16,211,194,196,190,204,196,198,198,197,189,195,197,193,194,183,110,158,71,253,247,40,111,26,3,14,16,234,45,214,210,250,86,132,237,90,98,
215,247,59,254,13,4,55,172,128,219,2,0,248,13,0,192,148,134,129,22,89,198,254,255,93,44,95,26,0,192,251,197,222,0,128,225,6,0,184,241,121,244,50,202,80,6,168,228,74,103,230,253,48,101,0,64,134,181,251,101,184,195,1,176,249,224,2,0,249,157,107,0,144,183,
2,0,157,78,0,96,229,2,64,95,221,116,2,192,151,48,125,242,4,188,190,181,62,0,220,17,224,9,128,156,114,22,66,175,108,106,219,187,147,39,25,85,200,205,144,86,177,191,177,240,204,87,222,189,76,146,141,242,251,138,33,130,21,8,56,87,164,41,136,129,19,5,250,
8,203,118,126,160,2,28,49,179,214,191,169,14,104,160,236,2,24,232,4,204,221,165,224,115,247,68,198,142,204,32,251,219,82,51,198,152,84,29,137,204,216,102,13,174,147,219,130,113,206,5,107,59,191,186,102,213,121,211,38,202,179,223,1,0,0,0,60,128,219,58,
210,48,208,162,78,30,199,100,0,192,135,29,0,244,219,246,135,0,0,224,98,115,249,78,238,235,42,5,0,24,123,252,223,255,22,1,0,64,178,189,119,207,22,0,241,193,38,0,16,223,188,1,0,190,25,0,72,121,0,72,121,0,200,229,47,239,1,192,202,3,64,255,26,0,72,58,0,47,
0,33,119,202,34,199,38,4,50,35,179,114,0,9,165,48,248,41,128,139,170,216,18,241,246,54,91,9,139,97,146,66,205,80,38,156,251,241,197,152,99,1,176,212,201,173,143,206,219,46,133,139,176,70,4,140,89,175,113,41,220,207,96,200,154,2,134,0,56,177,6,183,217,
61,130,137,161,205,70,197,167,191,13,109,27,93,82,14,7,110,180,219,212,125,230,3,116,15,103,106,214,247,43,29,211,161,215,207,133,139,14,192,63,243,202,43,0,48,252,230,111,248,8,0,134,175,116,171,48,13,89,16,157,250,25,97,83,209,2,0,200,109,150,25,0,
0,206,105,56,27,0,0,120,135,43,0,122,249,11,0,248,137,117,0,64,185,220,1,0,41,3,38,1,144,247,119,0,208,55,0,112,201,123,0,160,83,6,0,58,3,128,114,25,8,255,228,118,238,73,196,151,183,188,211,104,47,75,237,181,63,4,225,145,69,25,222,58,84,73,88,177,218,
249,91,66,34,164,2,90,200,91,135,66,81,121,79,59,52,86,174,83,110,141,164,144,140,176,221,50,152,164,68,10,171,83,31,224,220,90,1,49,212,6,199,34,140,160,174,9,62,63,132,1,199,126,127,159,216,32,128,196,151,203,7,0,174,147,235,237,189,109,150,236,157,
196,191,18,246,241,29,77,196,57,217,63,0,128,14,128,225,211,95,241,84,224,185,140,166,129,150,217,255,3,0,190,221,45,0,0,232,165,205,223,37,0,0,180,55,89,210,150,47,190,2,0,176,74,122,207,6,0,250,167,34,0,52,127,189,22,0,233,36,1,138,28,79,20,0,125,235,
181,5,32,239,45,0,189,188,67,1,210,73,132,196,166,12,72,185,195,166,152,190,148,137,12,202,16,33,187,28,197,125,22,51,104,164,96,190,232,173,15,151,209,193,247,135,140,242,144,172,137,18,177,81,190,129,109,41,157,58,213,222,86,233,149,38,173,120,193,
159,160,83,177,151,180,146,55,148,9,132,207,60,128,181,171,29,15,53,20,61,78,38,147,53,99,203,94,46,16,3,40,141,169,248,0,46,180,219,248,115,98,68,109,4,221,159,208,247,112,214,38,210,201,63,0,192,111,0,0,159,246,253,202,48,0,192,43,37,192,150,82,65,
139,44,61,251,179,3,0,184,249,222,0,0,88,254,217,121,0,0,96,24,93,8,0,0,176,255,101,239,182,40,0,64,243,216,104,63,88,7,0,141,254,245,10,0,174,229,0,128,116,30,0,36,0,114,58,229,114,0,200,114,103,0,57,191,95,91,0,24,129,57,64,76,249,114,214,250,23,6,
196,106,36,33,164,147,32,59,198,172,62,243,61,103,171,247,78,205,218,166,166,244,3,193,72,102,44,38,241,2,198,123,223,219,214,80,109,229,77,134,34,86,218,13,71,218,132,224,175,197,162,193,157,107,128,69,152,182,233,227,113,168,151,165,28,153,1,111,65,
55,204,16,112,157,245,239,254,74,101,239,2,117,142,105,39,56,189,201,1,238,243,91,149,165,212,220,105,73,254,151,98,91,247,227,236,219,136,165,103,127,0,0,183,3,143,223,1,0,0,0,60,15,192,84,76,163,168,83,236,179,3,0,110,254,63,4,0,0,92,254,125,8,0,184,
1,0,116,134,0,0,0,176,90,198,124,1,0,64,22,254,53,183,0,0,0,99,198,220,105,1,0,0,128,179,31,20,0,218,191,6,0,36,239,0,64,202,128,49,111,10,0,250,30,0,72,243,6,0,120,7,128,93,6,27,130,157,109,210,86,71,103,56,103,78,39,97,53,34,104,141,169,210,252,5,0,
41,208,201,255,17,161,125,22,243,65,225,189,33,48,45,15,70,225,34,150,27,152,128,19,32,25,105,248,207,97,156,104,193,173,195,167,166,216,6,149,131,0,43,29,68,153,0,228,128,76,34,178,88,130,96,16,117,53,116,5,238,179,219,213,165,100,195,28,234,159,184,
46,186,223,99,54,145,225,167,187,24,0,240,207,252,230,0,0,254,185,159,254,105,0,112,241,155,151,0,128,99,43,134,16,9,75,207,38,207,166,208,8,0,64,250,39,59,2,0,192,231,236,109,0,0,192,52,13,0,0,0,167,31,20,0,124,104,19,0,224,3,0,64,223,10,0,237,156,1,
144,188,21,0,218,191,6,0,122,202,3,128,59,1,128,64,237,220,9,196,240,28,63,196,93,85,97,56,55,72,235,143,160,237,140,103,236,222,20,175,124,124,56,31,22,23,40,167,26,17,41,250,158,116,168,80,205,52,116,62,145,102,117,24,202,98,126,146,221,5,182,28,235,
32,82,186,89,66,23,195,64,3,32,184,12,202,193,200,48,217,208,203,48,91,142,172,100,59,81,179,206,135,67,253,66,173,157,250,1,238,179,59,90,227,36,183,170,77,103,106,90,247,12,150,9,127,246,203,112,57,12,0,122,255,103,0,0,252,124,26,0,244,254,149,14,0,
6,83,42,26,69,81,194,8,71,206,62,77,246,0,0,248,191,29,0,192,45,22,0,0,168,109,179,111,0,0,192,28,71,1,0,160,101,44,111,0,32,239,127,77,1,224,173,60,0,190,83,6,0,233,12,0,119,242,151,6,0,233,53,5,208,94,223,61,24,40,124,50,34,144,232,24,172,213,67,32,
113,231,109,82,185,3,126,176,93,101,247,193,32,208,47,226,102,20,30,37,55,217,88,48,128,63,236,210,140,133,115,51,157,84,32,18,185,45,9,32,36,12,88,49,75,113,192,145,13,133,6,231,97,6,216,10,36,105,83,216,29,200,72,175,118,221,14,74,41,24,175,187,111,
79,109,113,1,141,228,97,0,174,115,91,213,101,146,150,204,147,116,127,66,223,51,172,154,14,111,255,0,0,67,7,128,151,255,220,167,1,0,134,79,43,90,107,20,69,179,220,135,207,0,128,15,178,111,0,0,36,206,121,2,0,0,63,90,228,223,42,1,0,24,35,209,126,104,93,
0,226,155,11,0,119,58,229,2,64,222,10,0,222,3,152,7,0,215,191,190,85,248,203,7,208,122,211,192,72,60,247,41,77,100,150,78,78,190,253,243,23,32,217,101,184,215,57,141,59,246,138,145,196,95,38,77,58,103,179,232,13,147,105,87,25,106,253,152,196,183,156,
112,159,159,218,236,113,117,35,20,210,101,24,45,151,161,180,67,35,27,220,249,33,81,2,70,30,11,177,49,76,40,21,8,209,177,17,44,35,161,54,245,99,89,159,58,98,171,106,255,234,173,109,225,35,184,96,58,57,110,51,219,184,105,134,68,115,6,241,45,228,126,147,
24,142,235,167,247,11,0,252,243,139,79,3,192,240,149,127,198,71,29,0,253,211,131,41,108,77,161,39,170,255,217,95,2,0,172,215,34,0,0,112,115,240,7,0,0,125,184,0,160,127,121,3,128,252,229,122,5,128,228,29,0,220,9,240,17,0,226,3,1,128,190,5,2,212,245,173,
41,66,52,96,1,224,13,8,109,74,196,72,135,124,198,214,251,121,177,217,185,69,170,122,52,141,159,113,223,122,203,207,166,80,183,244,34,123,56,198,18,146,191,116,80,21,41,103,43,64,24,23,42,113,121,169,216,117,250,80,91,139,160,199,163,14,58,191,37,137,
106,169,29,73,8,114,146,42,198,13,109,161,4,103,26,21,158,82,108,180,151,146,165,53,2,101,238,186,1,174,51,219,136,105,50,146,237,29,116,95,33,189,71,107,162,94,253,3,0,24,0,224,211,255,220,177,74,197,16,121,178,220,207,178,29,0,248,17,0,0,192,242,102,
78,238,110,11,0,0,171,165,229,39,158,6,160,31,124,232,191,220,20,0,241,205,0,160,31,68,0,64,167,28,0,56,229,0,192,229,0,192,157,114,0,224,242,215,22,0,125,3,0,46,11,0,161,28,135,150,227,221,60,167,232,134,177,78,216,145,63,40,84,54,109,140,157,101,185,
99,247,153,142,23,217,205,199,74,44,22,99,43,213,191,17,23,105,187,175,34,217,160,116,217,186,124,220,186,27,23,129,104,66,119,249,211,225,7,29,153,102,22,167,32,131,148,192,136,17,6,150,148,174,169,49,205,150,67,89,67,198,160,38,146,140,82,211,3,60,
145,85,33,96,156,30,238,82,91,237,121,114,65,194,51,235,159,66,238,227,120,13,247,237,23,46,59,0,253,43,0,0,62,94,2,0,191,97,72,67,8,148,135,37,58,153,157,197,252,3,0,112,120,3,0,184,57,2,0,0,180,105,81,5,0,128,108,246,216,37,1,32,239,127,13,0,228,250,
62,0,40,119,6,0,233,60,0,148,1,160,249,235,229,18,0,68,103,0,164,83,46,39,144,106,1,64,238,200,113,121,173,226,240,58,192,116,92,78,108,238,4,61,226,85,204,69,31,102,20,27,220,206,101,10,214,85,92,51,251,157,9,29,11,235,151,162,248,171,74,58,91,177,212,
72,173,40,145,166,117,0,96,14,131,41,172,147,205,55,195,166,87,49,173,125,1,64,56,23,196,53,199,14,66,96,72,35,42,135,49,100,11,47,214,21,119,138,198,5,106,225,124,9,0,174,83,219,201,169,148,82,180,17,228,63,209,190,123,6,218,116,232,236,151,75,189,3,
224,43,0,0,158,219,2,151,105,210,12,84,8,198,136,126,246,217,27,3,0,120,103,183,0,64,255,72,170,3,0,0,234,180,191,74,11,0,0,100,115,171,158,250,80,1,16,151,222,222,2,128,188,247,45,0,224,175,0,128,119,0,32,86,184,111,86,0,168,183,4,128,94,58,29,0,176,
7,151,50,30,96,101,148,59,64,24,253,74,101,176,165,140,147,71,245,226,27,10,214,76,171,158,129,143,26,24,50,31,103,214,132,131,64,198,169,121,75,146,6,209,79,99,17,67,96,48,52,245,52,237,67,194,228,128,203,75,16,62,138,219,34,220,64,215,236,253,129,135,
106,138,72,39,156,105,71,246,74,1,117,68,182,111,56,146,0,16,118,234,7,110,51,59,113,189,52,202,76,242,17,219,190,167,32,27,174,213,143,238,18,0,95,121,229,37,0,232,254,0,0,42,128,41,41,99,158,51,106,194,62,184,49,0,128,57,167,61,0,0,160,243,111,59,0,
112,121,33,1,0,0,68,51,71,5,0,0,114,171,46,127,13,0,184,252,53,0,32,29,0,32,157,0,128,18,116,255,242,158,0,128,188,3,128,78,32,0,115,202,96,178,1,130,154,242,42,167,140,73,116,8,35,190,64,241,33,255,87,119,112,214,194,47,237,93,169,148,34,149,8,118,204,
51,251,177,76,241,229,15,66,102,35,73,80,111,137,246,22,141,233,213,37,123,173,143,98,240,167,88,0,38,118,223,249,202,102,62,74,205,105,11,203,56,83,9,162,149,61,73,211,37,146,175,236,151,116,43,31,37,199,90,220,90,86,0,110,115,122,163,126,178,32,95,
186,65,131,98,66,95,175,10,220,94,2,165,141,86,68,244,147,235,239,247,205,37,3,0,116,70,151,197,118,12,0,22,238,236,93,94,250,230,94,90,67,193,217,247,254,229,175,161,46,87,185,190,93,103,148,183,242,203,1,178,156,176,201,11,33,159,129,99,229,123,241,
158,20,164,225,178,65,230,44,148,243,115,232,116,50,157,153,27,219,55,116,202,150,33,199,189,229,114,30,111,70,180,110,210,169,156,243,214,155,82,124,186,186,44,220,106,187,52,22,139,109,121,164,135,109,15,59,237,126,175,223,25,137,97,199,161,113,76,
10,231,101,114,4,106,154,98,53,188,167,210,40,42,50,231,160,33,216,79,200,73,191,213,212,38,191,215,180,13,6,12,4,133,56,114,1,46,175,250,103,73,119,105,130,177,184,206,212,62,27,16,126,83,2,20,35,7,0,132,24,66,12,49,6,24,67,216,203,213,229,221,219,213,
219,187,183,119,31,124,200,217,94,122,202,211,223,252,180,167,126,226,3,215,55,0,239,127,185,186,92,229,122,121,191,212,229,122,161,151,78,8,0,128,78,25,0,30,175,83,238,128,28,178,222,23,232,148,1,0,4,160,15,0,1,1,192,65,0,232,4,8,64,3,208,208,0,0,0 };

const char* singing_ogg = (const char*) audioFileData;
